<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Service;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\mcp_server\Entity\McpToolConfig;

/**
 * Discovers and aggregates OAuth scopes from MCP tool configurations.
 *
 * This service provides metadata discovery for OAuth scopes supported by the
 * MCP server based on enabled tool configurations. Scopes are aggregated,
 * and deduplicated.
 */
final class OAuthScopeDiscoveryService {

  /**
   * Constructs a new OAuthScopeDiscoveryService.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager for loading MCP tool configurations and scopes.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Gets all unique OAuth scopes supported by enabled MCP tools.
   *
   * This method:
   * - Loads all enabled McpToolConfig entities
   * - Extracts scopes from each configuration
   * - Deduplicates and sorts scopes alphabetically.
   *
   * @return array
   *   Array of unique scope IDs (strings), sorted alphabetically.
   *   Empty array if no scopes are configured.
   */
  public function getScopesSupported(): array {
    // Load all enabled MCP tool configurations.
    try {
      $storage = $this->entityTypeManager->getStorage('mcp_tool_config');
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException  $e) {
      return [];
    }
    $config_ids = $storage->getQuery()
      ->condition('status', TRUE)
      ->accessCheck()
      ->execute();

    if (empty($config_ids)) {
      return [];
    }

    // Load all enabled configurations.
    $configs = $storage->loadMultiple($config_ids);

    // Aggregate all scope names from configurations.
    // Note: getScopes() returns scope names (strings), not entity IDs.
    $all_scope_names = array_reduce(
      $configs,
      static function (array $scopes, $config): array {
        assert($config instanceof McpToolConfig);
        return array_merge($scopes, $config->getScopeNames());
      },
      []
    );

    // Deduplicate and sort scope names alphabetically.
    $scope_names = array_values(array_unique($all_scope_names));
    sort($scope_names);

    return $scope_names;
  }

}
