<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\simple_oauth\Plugin\Field\FieldType\Oauth2ScopeReferenceItemInterface;
use Drupal\simple_oauth\Server\ResourceServerFactoryInterface;
use Symfony\Bridge\PsrHttpMessage\HttpFoundationFactoryInterface;
use Symfony\Bridge\PsrHttpMessage\HttpMessageFactoryInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Validates OAuth scopes from access tokens.
 *
 * Follows the pattern from jsonrpc_mcp module for extracting and validating
 * OAuth scopes from Simple OAuth tokens. This service provides defensive
 * programming by returning empty arrays on any failure condition.
 */
final class OAuthScopeValidator {

  /**
   * Constructs a new OAuthScopeValidator.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager for loading oauth2_token and oauth2_scope
   *   entities.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack for accessing the current request.
   * @param \Symfony\Bridge\PsrHttpMessage\HttpMessageFactoryInterface $httpMessageFactory
   *   The PSR-7 HTTP message factory.
   * @param \Drupal\simple_oauth\Server\ResourceServerFactoryInterface $resourceServerFactory
   *   The OAuth2 resource server factory.
   * @param \Symfony\Bridge\PsrHttpMessage\HttpFoundationFactoryInterface $httpFoundationFactory
   *   The HTTP foundation factory.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly RequestStack $requestStack,
    private readonly HttpMessageFactoryInterface $httpMessageFactory,
    private readonly ResourceServerFactoryInterface $resourceServerFactory,
    private readonly HttpFoundationFactoryInterface $httpFoundationFactory,
  ) {}

  /**
   * Extracts OAuth scopes from the current request's access token.
   *
   * This method:
   * - Validates the Bearer token using Simple OAuth's resource server
   * - Loads the corresponding oauth2_token entity by oauth_access_token_id
   * - Validates the token is not revoked
   * - Loads scope entities and extracts their names.
   *
   * @return array
   *   Array of scope name strings (e.g., "tutorial:read"). Empty array if
   *   no token, invalid token, or if any error occurs during extraction.
   */
  public function extractTokenScopes(): array {
    // Get the current request.
    $request = $this->requestStack->getCurrentRequest();
    if (!$request) {
      return [];
    }

    // Check for Bearer token in Authorization header.
    $authorization = $request->headers->get('Authorization');
    if (!$authorization || !str_starts_with($authorization, 'Bearer ')) {
      return [];
    }

    try {
      // Create a PSR-7 message from the request for OAuth library validation.
      $psr7_request = $this->httpMessageFactory->createRequest($request);
      $resource_server = $this->resourceServerFactory->get();
      $output_psr7_request = $resource_server->validateAuthenticatedRequest($psr7_request);

      // Convert back to HttpFoundation request to access attributes.
      $auth_request = $this->httpFoundationFactory->createRequest($output_psr7_request);

      // Load oauth2_token entity by oauth_access_token_id.
      $token_storage = $this->entityTypeManager->getStorage('oauth2_token');
      $tokens = $token_storage->loadByProperties([
        'value' => $auth_request->attributes->get('oauth_access_token_id'),
      ]);

      if (empty($tokens)) {
        return [];
      }

      /** @var \Drupal\simple_oauth\Entity\Oauth2TokenInterface $token */
      $token = reset($tokens);

      // Check if token is revoked.
      if ($token->isRevoked()) {
        return [];
      }

      // Extract scope IDs from token field values.
      $field_item_list = $token->get('scopes');
      $scope_ids = [];
      foreach ($field_item_list as $field_item) {
        assert($field_item instanceof Oauth2ScopeReferenceItemInterface);
        $scope_ids[] = $field_item->getScope()->getName();
      }
      return $scope_ids;
    }
    catch (\Exception $e) {
      // Token validation or loading failed, return empty array.
      return [];
    }
  }

  /**
   * Validates that token scopes include all required scopes.
   *
   * This method uses AND logic: ALL required scopes must be present in the
   * token scopes for validation to succeed.
   *
   * @param array $required_scopes
   *   Array of required scope IDs (strings).
   * @param array $token_scopes
   *   Array of scope IDs from the token (strings).
   *
   * @return \Drupal\mcp_server\Service\ScopeValidationResult
   *   Validation result with details about missing scopes.
   */
  public function validateScopes(array $required_scopes, array $token_scopes): ScopeValidationResult {
    // Calculate missing scopes using array_diff (required - token).
    $missing_scopes = array_diff($required_scopes, $token_scopes);

    // Validation succeeds if there are no missing scopes.
    $is_valid = empty($missing_scopes);

    return new ScopeValidationResult(
      isValid: $is_valid,
      missingScopes: array_values($missing_scopes),
      requiredScopes: $required_scopes,
      tokenScopes: $token_scopes,
    );
  }

}
