<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Session;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Mcp\Server\Session\SessionStoreInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Uid\Uuid;

/**
 * Database-backed session store for MCP Server.
 *
 * Provides persistent session storage using Drupal's database abstraction
 * layer. Session data is stored in the mcp_session_queue table with automatic
 * expiry management. This store supports distributed deployments and handles
 * large session data efficiently using BLOB storage.
 */
final class DatabaseSessionStore implements SessionStoreInterface {

  /**
   * Constructs a DatabaseSessionStore.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly Connection $database,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function exists(Uuid $id): bool {
    try {
      $result = $this->database->select('mcp_session_queue', 's')
        ->fields('s', ['session_id'])
        ->condition('session_id', $id->toRfc4122())
        ->condition('expires_at', time(), '>')
        ->execute()
        ->fetchField();

      return $result !== FALSE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to check session existence: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function read(Uuid $id): string|false {
    try {
      $result = $this->database->select('mcp_session_queue', 's')
        ->fields('s', ['data'])
        ->condition('session_id', $id->toRfc4122())
        ->condition('expires_at', time(), '>')
        ->execute()
        ->fetchField();

      if ($result === FALSE) {
        return FALSE;
      }

      return is_resource($result) ? stream_get_contents($result) : $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to read session: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function write(Uuid $id, string $data): bool {
    try {
      $ttl = (int) ($this->configFactory->get('mcp_server.settings')
        ->get('session.ttl') ?? 86400);

      $this->database->merge('mcp_session_queue')
        ->key('session_id', $id->toRfc4122())
        ->fields([
          'data' => $data,
          'last_activity' => time(),
          'expires_at' => time() + $ttl,
        ])
        ->execute();

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to write session: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function destroy(Uuid $id): bool {
    try {
      $deleted = $this->database->delete('mcp_session_queue')
        ->condition('session_id', $id->toRfc4122())
        ->execute();

      return $deleted > 0;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to destroy session: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function gc(): array {
    try {
      $query = $this->database->select('mcp_session_queue', 's')
        ->fields('s', ['session_id'])
        ->condition('expires_at', time(), '<=');

      $expired_ids = $query->execute()->fetchCol();

      if (!empty($expired_ids)) {
        $this->database->delete('mcp_session_queue')
          ->condition('session_id', $expired_ids, 'IN')
          ->execute();

        $this->logger->info('Garbage collected @count expired sessions', [
          '@count' => count($expired_ids),
        ]);

        return array_map(
          fn(string $id) => Uuid::fromString($id),
          $expired_ids
        );
      }

      return [];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to garbage collect sessions: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

}
