<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Session;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Psr\Log\LoggerInterface;
use Symfony\Component\Uid\Uuid;

/**
 * Database-backed session manager for MCP Server.
 *
 * Manages MCP session lifecycle including creation, validation, and
 * destruction. Sessions are stored in the mcp_session_metadata table with
 * filesystem roots and expiry tracking. This manager handles persistent
 * sessions that survive server restarts, unlike JWT-based sessions.
 */
final class DbSessionManager {

  /**
   * Constructs a DbSessionManager.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly Connection $database,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * Creates a new session with filesystem roots.
   *
   * Generates a cryptographically secure UUID v4 session identifier and stores
   * session metadata in the database. The session will expire according to the
   * configured TTL.
   *
   * @param array $roots
   *   Allowed filesystem roots for file operations. Defaults to empty array.
   *
   * @return string
   *   The session ID (UUID v4 format).
   *
   * @throws \Exception
   *   If session creation fails.
   */
  public function createSession(array $roots = []): string {
    try {
      $session_id = Uuid::v4()->toRfc4122();
      $ttl = (int) ($this->configFactory->get('mcp_server.settings')
        ->get('session.ttl') ?? 86400);
      $now = time();

      $this->database->insert('mcp_session_metadata')
        ->fields([
          'session_id' => $session_id,
          'roots' => json_encode($roots),
          'created_at' => $now,
          'expires_at' => $now + $ttl,
          'last_activity' => $now,
        ])
        ->execute();

      $this->logger->info('Created session @session_id with TTL @ttl seconds', [
        '@session_id' => $session_id,
        '@ttl' => $ttl,
      ]);

      return $session_id;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create session: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

  /**
   * Validates a session and returns its context.
   *
   * Queries the database for session metadata, validates expiry, and
   * constructs a SessionContext object. This method does not update the
   * last_activity timestamp.
   *
   * @param string $session_id
   *   The session ID to validate.
   *
   * @return \Drupal\mcp_server\Session\SessionContext
   *   The session context with metadata.
   *
   * @throws \Drupal\mcp_server\Session\SessionValidationException
   *   If session is not found, expired, or validation fails.
   */
  public function validateSession(string $session_id): SessionContext {
    try {
      $result = $this->database->select('mcp_session_metadata', 's')
        ->fields('s', ['session_id', 'roots', 'expires_at'])
        ->condition('session_id', $session_id)
        ->execute()
        ->fetchAssoc();

      if (!$result) {
        throw new SessionValidationException(
          'Session not found',
          404
        );
      }

      $expiry = (int) $result['expires_at'];
      if ($expiry <= time()) {
        throw new SessionValidationException(
          'Session has expired',
          401
        );
      }

      $roots = json_decode($result['roots'], TRUE);
      if (!is_array($roots)) {
        throw new SessionValidationException(
          'Invalid session data',
          500
        );
      }

      return new SessionContext(
        sessionId: $session_id,
        expiry: $expiry,
        roots: $roots,
      );
    }
    catch (SessionValidationException $e) {
      throw $e;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to validate session: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw new SessionValidationException(
        'Session validation failed',
        500,
        $e
      );
    }
  }

  /**
   * Updates the last activity timestamp for a session.
   *
   * This should be called on each request to extend the session lifetime and
   * prevent premature garbage collection. The expiry timestamp is also updated
   * based on the configured TTL.
   *
   * @param string $session_id
   *   The session ID to update.
   *
   * @throws \Exception
   *   If update fails.
   */
  public function updateActivity(string $session_id): void {
    try {
      $ttl = (int) ($this->configFactory->get('mcp_server.settings')
        ->get('session.ttl') ?? 86400);
      $now = time();

      $updated = $this->database->update('mcp_session_metadata')
        ->fields([
          'last_activity' => $now,
          'expires_at' => $now + $ttl,
        ])
        ->condition('session_id', $session_id)
        ->execute();

      if ($updated === 0) {
        $this->logger->warning('Attempted to update non-existent session: @session_id', [
          '@session_id' => $session_id,
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to update session activity: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

  /**
   * Destroys a session and removes all associated data.
   *
   * Deletes the session from mcp_session_metadata table. Associated session
   * queue data should be removed separately via DatabaseSessionStore.
   *
   * @param string $session_id
   *   The session ID to destroy.
   *
   * @return bool
   *   TRUE if session was deleted, FALSE if session did not exist.
   *
   * @throws \Exception
   *   If deletion fails.
   */
  public function destroySession(string $session_id): bool {
    try {
      $deleted = $this->database->delete('mcp_session_metadata')
        ->condition('session_id', $session_id)
        ->execute();

      if ($deleted > 0) {
        $this->logger->info('Destroyed session @session_id', [
          '@session_id' => $session_id,
        ]);
        return TRUE;
      }

      return FALSE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to destroy session: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

  /**
   * Garbage collects expired sessions.
   *
   * Removes all sessions where expires_at is in the past. Returns the count of
   * removed sessions for monitoring and logging purposes.
   *
   * @return int
   *   The number of expired sessions removed.
   *
   * @throws \Exception
   *   If garbage collection fails.
   */
  public function gc(): int {
    try {
      $deleted = $this->database->delete('mcp_session_metadata')
        ->condition('expires_at', time(), '<=')
        ->execute();

      if ($deleted > 0) {
        $this->logger->info('Garbage collected @count expired sessions', [
          '@count' => $deleted,
        ]);
      }

      return $deleted;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to garbage collect sessions: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

}
