<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Session;

/**
 * Value object representing MCP session context.
 *
 * This object carries session information including filesystem roots.
 * It provides a type-safe interface for accessing session information.
 */
final readonly class SessionContext {

  /**
   * Constructs a SessionContext.
   *
   * @param string $sessionId
   *   Unique session identifier (UUID v4).
   * @param int $expiry
   *   Unix timestamp when session expires.
   * @param array $roots
   *   Allowed filesystem roots for file operations.
   */
  public function __construct(
    public string $sessionId,
    public int $expiry,
    public array $roots,
  ) {}

  /**
   * Checks if session is expired.
   *
   * @return bool
   *   TRUE if session has expired, FALSE otherwise.
   */
  public function isExpired(): bool {
    return time() >= $this->expiry;
  }

  /**
   * Checks if a filesystem root is allowed.
   *
   * @param string $root
   *   Root path to check.
   *
   * @return bool
   *   TRUE if root is allowed, FALSE otherwise.
   */
  public function isRootAllowed(string $root): bool {
    return in_array($root, $this->roots, TRUE);
  }

}
