<?php

declare(strict_types=1);

namespace Drupal\mcp_server;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\tool\Tool\ToolDefinition;
use Drupal\tool\Tool\ToolInterface;
use Drupal\tool\Tool\ToolManager;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Mcp\Server\ClientGateway;

/**
 * Service for discovering and integrating Tool API tools.
 */
final class ToolApiDiscovery {

  /**
   * Constructs a ToolApiDiscovery service.
   *
   * @param \Drupal\tool\Tool\ToolManager $toolManager
   *   The Tool API plugin manager.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(
    private readonly ToolManager $toolManager,
    private readonly LoggerChannelInterface $logger,
  ) {}

  /**
   * Gets all available Tool API tools.
   *
   * @return array<string, array>
   *   Array of tool metadata keyed by tool ID. Each tool contains:
   *   - id: string - The tool plugin ID
   *   - label: string - Human-readable label
   *   - description: string - Tool description
   */
  public function getAvailableTools(): array {
    try {
      $definitions = $this->toolManager->getDefinitions();
      $tools = [];

      foreach ($definitions as $plugin_id => $definition) {
        if (!$definition instanceof ToolDefinition) {
          continue;
        }

        $tools[$plugin_id] = [
          'id' => $plugin_id,
          'label' => (string) $definition->getLabel(),
          'description' => (string) $definition->getDescription(),
        ];
      }

      return $tools;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve Tool API tools: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Gets a specific tool definition.
   *
   * @param string $toolId
   *   The tool plugin ID.
   *
   * @return array|null
   *   Tool definition array with metadata and schema, or NULL if not found.
   */
  public function getToolDefinition(string $toolId): ?array {
    try {
      $definition = $this->toolManager->getDefinition($toolId, FALSE);

      if (!$definition instanceof ToolDefinition) {
        return NULL;
      }

      return [
        'id' => $toolId,
        'label' => (string) $definition->getLabel(),
        'description' => (string) $definition->getDescription(),
        'inputSchema' => $this->convertToMcpSchema($definition),
        'destructive' => $definition->isDestructive(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve tool @tool: @message', [
        '@tool' => $toolId,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Converts Tool API schema to MCP parameter schema.
   *
   * @param \Drupal\tool\Tool\ToolDefinition $definition
   *   Tool API tool definition.
   *
   * @return array
   *   MCP-compatible parameter schema in JSON Schema format.
   */
  public function convertToMcpSchema(ToolDefinition $definition): array {
    $properties = [];
    $required = [];

    $input_definitions = $definition->getInputDefinitions();

    foreach ($input_definitions as $name => $input_definition) {
      $property_schema = $this->convertInputDefinitionToSchema(
        $input_definition
      );
      $properties[$name] = $property_schema;

      if ($input_definition->isRequired()) {
        $required[] = $name;
      }
    }

    $schema = [
      'type' => 'object',
      'properties' => $properties,
    ];

    if (!empty($required)) {
      $schema['required'] = $required;
    }

    return $schema;
  }

  /**
   * Converts an InputDefinition to a JSON Schema property.
   *
   * @param \Drupal\tool\TypedData\InputDefinitionInterface $definition
   *   The input definition.
   *
   * @return array
   *   JSON Schema property definition.
   */
  private function convertInputDefinitionToSchema(
    InputDefinitionInterface $definition,
  ): array {
    $data_type = $definition->getDataType();
    $schema = [];

    // Map Drupal typed data types to JSON Schema types.
    $type_map = [
      'string' => 'string',
      'integer' => 'integer',
      'float' => 'number',
      'boolean' => 'boolean',
      'email' => 'string',
      'uri' => 'string',
      'datetime_iso8601' => 'string',
      'timestamp' => 'integer',
      'list' => 'array',
      'map' => 'object',
    ];

    // Handle entity reference types.
    if (str_starts_with($data_type, 'entity:') ||
        str_starts_with($data_type, 'entity_reference:')) {
      $schema['type'] = 'string';
      $schema['description'] = 'Entity ID or UUID';
    }
    elseif (isset($type_map[$data_type])) {
      $schema['type'] = $type_map[$data_type];
    }
    else {
      // Default to string for unknown types.
      $schema['type'] = 'string';
      $this->logger->warning(
        'Unknown data type @type mapped to string',
        ['@type' => $data_type]
      );
    }

    // Add description if available.
    $description = $definition->getDescription();
    if ($description) {
      $schema['description'] = (string) $description;
    }

    // Add constraints as additional schema properties.
    $constraints = $definition->getConstraints();
    foreach ($constraints as $constraint_name => $constraint_config) {
      match ($constraint_name) {
        'Length' => $this->addLengthConstraints(
          $schema,
          $constraint_config
        ),
        'Range' => $this->addRangeConstraints($schema, $constraint_config),
        'Regex' => $this->addRegexConstraint($schema, $constraint_config),
        'AllowedValues' => $this->addEnumConstraint(
          $schema,
          $constraint_config
        ),
        default => NULL,
      };
    }

    return $schema;
  }

  /**
   * Adds length constraints to schema.
   *
   * @param array &$schema
   *   The schema array to modify.
   * @param array|object $constraint
   *   The constraint configuration.
   */
  private function addLengthConstraints(
    array &$schema,
    array|object $constraint,
  ): void {
    $min = is_array($constraint) ? ($constraint['min'] ?? NULL) :
      ($constraint->min ?? NULL);
    $max = is_array($constraint) ? ($constraint['max'] ?? NULL) :
      ($constraint->max ?? NULL);

    if ($min !== NULL) {
      $schema['minLength'] = $min;
    }
    if ($max !== NULL) {
      $schema['maxLength'] = $max;
    }
  }

  /**
   * Adds range constraints to schema.
   *
   * @param array &$schema
   *   The schema array to modify.
   * @param array|object $constraint
   *   The constraint configuration.
   */
  private function addRangeConstraints(
    array &$schema,
    array|object $constraint,
  ): void {
    $min = is_array($constraint) ? ($constraint['min'] ?? NULL) :
      ($constraint->min ?? NULL);
    $max = is_array($constraint) ? ($constraint['max'] ?? NULL) :
      ($constraint->max ?? NULL);

    if ($min !== NULL) {
      $schema['minimum'] = $min;
    }
    if ($max !== NULL) {
      $schema['maximum'] = $max;
    }
  }

  /**
   * Adds regex pattern constraint to schema.
   *
   * @param array &$schema
   *   The schema array to modify.
   * @param array|object $constraint
   *   The constraint configuration.
   */
  private function addRegexConstraint(
    array &$schema,
    array|object $constraint,
  ): void {
    $pattern = is_array($constraint) ? ($constraint['pattern'] ?? NULL) :
      ($constraint->pattern ?? NULL);

    if ($pattern !== NULL) {
      $schema['pattern'] = $pattern;
    }
  }

  /**
   * Adds enum constraint to schema.
   *
   * @param array &$schema
   *   The schema array to modify.
   * @param array|object $constraint
   *   The constraint configuration.
   */
  private function addEnumConstraint(
    array &$schema,
    array|object $constraint,
  ): void {
    $choices = is_array($constraint) ? ($constraint['choices'] ?? NULL) :
      ($constraint->choices ?? NULL);

    if (is_array($choices) && !empty($choices)) {
      $schema['enum'] = array_values($choices);
    }
  }

  /**
   * Executes a Tool API tool.
   *
   * @param string $toolId
   *   The tool plugin ID.
   * @param array $parameters
   *   Tool parameters keyed by input name.
   * @param \Mcp\Server\ClientGateway|null $gateway
   *   Optional client gateway for tools that support sampling.
   *
   * @return mixed
   *   Tool execution result.
   *
   * @throws \RuntimeException
   *   If tool execution fails.
   */
  public function executeTool(
    string $toolId,
    array $parameters,
    ?ClientGateway $gateway = NULL,
  ): mixed {
    try {
      // Create an instance of the tool plugin.
      $tool = $this->toolManager->createInstance($toolId);

      if (!$tool instanceof ToolInterface) {
        throw new \RuntimeException(
          "Tool plugin $toolId does not implement ToolInterface"
        );
      }

      // Inject gateway if tool implements ClientGatewayAwareInterface.
      if ($gateway !== NULL && $tool instanceof ClientGatewayAwareInterface) {
        $tool->setClientGateway($gateway);
      }

      // Set each parameter using setInputValue().
      foreach ($parameters as $name => $value) {
        $tool->setInputValue($name, $value);
      }

      // Execute the tool (no parameters needed).
      $tool->execute();

      // Get the result.
      $result = $tool->getResult();

      return [
        'success' => $result->isSuccess(),
        'message' => (string) $result->getMessage(),
        'data' => $result->getContextValues(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Tool execution failed for @tool: @message', [
        '@tool' => $toolId,
        '@message' => $e->getMessage(),
      ]);
      throw new \RuntimeException(
        "Tool execution failed: {$e->getMessage()}",
        0,
        $e
      );
    }
  }

}
