<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Traits;

use Mcp\Schema\Content\Content;
use Mcp\Schema\Result\CreateSamplingMessageResult;
use Mcp\Server\ClientGateway;

/**
 * Provides sampling capabilities for MCP tool plugins.
 *
 * Tools that need to request LLM sampling from the client should implement
 * ClientGatewayAwareInterface and can use this trait for convenient
 * implementation.
 *
 * @see \Drupal\mcp_server\ClientGatewayAwareInterface
 */
trait McpToolSamplingTrait {

  /**
   * The client gateway for communicating with the MCP client.
   */
  protected ?ClientGateway $clientGateway = NULL;

  /**
   * Sets the client gateway for sampling requests.
   *
   * @param \Mcp\Server\ClientGateway $gateway
   *   The client gateway instance.
   */
  public function setClientGateway(ClientGateway $gateway): void {
    $this->clientGateway = $gateway;
  }

  /**
   * Requests LLM sampling from the client.
   *
   * @param array<\Mcp\Schema\Content\SamplingMessage>|\Mcp\Schema\Content\Content|string $message
   *   The message to send for sampling. Can be a string, Content object,
   *   or array of SamplingMessage objects.
   * @param int $max_tokens
   *   Maximum tokens to generate.
   * @param int $timeout
   *   Timeout in seconds for the sampling request.
   * @param array<string, mixed> $options
   *   Additional sampling options (preferences, systemPrompt, temperature,
   *   includeContext, stopSequences, metadata).
   *
   * @return \Mcp\Schema\Result\CreateSamplingMessageResult
   *   The sampling response from the client.
   *
   * @throws \RuntimeException
   *   If the client gateway has not been set.
   */
  protected function sample(
    array|Content|string $message,
    int $max_tokens = 1000,
    int $timeout = 120,
    array $options = [],
  ): CreateSamplingMessageResult {
    if ($this->clientGateway === NULL) {
      throw new \RuntimeException('ClientGateway not available for sampling');
    }
    return $this->clientGateway->sample($message, $max_tokens, $timeout, $options);
  }

}
