<?php

declare(strict_types=1);

namespace Drupal\mcp_server_test\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mcp_server\ClientGatewayAwareInterface;
use Drupal\mcp_server\Traits\McpToolSamplingTrait;
use Drupal\node\NodeInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tool that generates node summaries using LLM sampling.
 *
 * Accepts either a full URL (path alias or /node/123) or a JSON:API URL
 * (e.g., /jsonapi/node/{bundle}/{uuid}), resolves it to a node entity,
 * extracts body content, uses MCP sampling to generate a summary, and stores
 * the result in the body.summary field.
 */
#[Tool(
  id: 'mcp_server_test:node_summary',
  label: new TranslatableMarkup('Node Summary Tool'),
  description: new TranslatableMarkup('Generates a summary for any node using LLM sampling'),
  operation: ToolOperation::Write,
  input_definitions: [
    'url' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup('URL'),
      description: new TranslatableMarkup('The full URL of the node to summarize (supports path aliases, /node/ID, or JSON:API URLs like /jsonapi/node/{bundle}/{uuid})'),
      required: TRUE,
    ),
  ],
)]
final class NodeSummaryTool extends ToolBase implements ClientGatewayAwareInterface {

  use McpToolSamplingTrait;

  /**
   * Maximum tokens for summary generation.
   */
  private const MAX_TOKENS = 500;

  /**
   * The alias manager for resolving path aliases.
   */
  protected AliasManagerInterface $aliasManager;

  /**
   * The entity type manager for loading entities.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->aliasManager = $container->get('path_alias.manager');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    $url = $values['url'] ?? '';

    if ($url === '') {
      return ExecutableResult::failure($this->t('URL parameter is required'));
    }

    $path = $this->extractPathFromUrl($url);
    if ($path === NULL) {
      return ExecutableResult::failure(
        $this->t('Invalid URL format: @url', ['@url' => $url])
      );
    }

    // Check if this is a JSON:API URL.
    $node = $this->resolveNodeFromJsonApiPath($path);
    if ($node === NULL) {
      // Try resolving as a regular path/alias.
      $internal_path = $this->aliasManager->getPathByAlias($path);
      $node_id = $this->extractNodeIdFromPath($internal_path);
      if ($node_id === NULL) {
        return ExecutableResult::failure(
          $this->t('URL does not resolve to a node: @path', ['@path' => $path])
        );
      }

      $node = $this->loadNodeById($node_id);
      if ($node === NULL) {
        return ExecutableResult::failure(
          $this->t('Node not found with ID: @id', ['@id' => $node_id])
        );
      }
    }

    if (!$node->hasField('body') || $node->get('body')->isEmpty()) {
      return ExecutableResult::failure(
        $this->t('Node does not have body content')
      );
    }

    $body_value = $node->get('body')->value;
    if (empty($body_value)) {
      return ExecutableResult::failure(
        $this->t('Node body field is empty')
      );
    }

    $prompt = $this->buildSummaryPrompt($body_value);

    try {
      $result = $this->sample($prompt, self::MAX_TOKENS);
      $summary = $result->content->text ?? '';

      if (empty($summary)) {
        return ExecutableResult::failure(
          $this->t('LLM returned empty summary')
        );
      }

      /** @var \Drupal\text\Plugin\Field\FieldType\TextWithSummaryItem $body_item */
      $body_item = $node->get('body')->first();
      $node->set('body', [
        'value' => $body_item->value,
        'summary' => $summary,
        'format' => $body_item->format,
      ]);
      $node->save();

      return ExecutableResult::success(
        $this->t('Summary generated and saved successfully'),
        [
          'node_id' => $node->id(),
          'bundle' => $node->bundle(),
          'summary' => $summary,
          'model' => $result->model,
        ]
      );
    }
    catch (\RuntimeException $e) {
      return ExecutableResult::failure(
        $this->t('Sampling failed: @message', ['@message' => $e->getMessage()])
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(
    array $values,
    AccountInterface $account,
    bool $return_as_object = FALSE,
  ): bool|AccessResultInterface {
    // Check generic node editing permission.
    $access_result = AccessResult::allowedIfHasPermission(
      $account,
      'administer nodes'
    )->orIf(
      AccessResult::allowedIfHasPermission($account, 'bypass node access')
    );
    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

  /**
   * Extracts the path component from a URL.
   *
   * @param string $url
   *   The full URL to parse.
   *
   * @return string|null
   *   The path component, or NULL if parsing fails.
   */
  private function extractPathFromUrl(string $url): ?string {
    $parsed = parse_url($url);
    if ($parsed === FALSE || !isset($parsed['path'])) {
      return NULL;
    }
    return $parsed['path'];
  }

  /**
   * Extracts the node ID from an internal path.
   *
   * @param string $path
   *   The internal path (e.g., /node/123).
   *
   * @return int|null
   *   The node ID, or NULL if the path is not a node path.
   */
  private function extractNodeIdFromPath(string $path): ?int {
    if (!preg_match('#^/node/(\d+)$#', $path, $matches)) {
      return NULL;
    }
    return (int) $matches[1];
  }

  /**
   * Loads a node entity by ID.
   *
   * @param int $node_id
   *   The node ID to load.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The node entity, or NULL if not found.
   */
  private function loadNodeById(int $node_id): ?NodeInterface {
    $node = $this->entityTypeManager->getStorage('node')->load($node_id);
    return $node instanceof NodeInterface ? $node : NULL;
  }

  /**
   * Resolves a node from a JSON:API path.
   *
   * @param string $path
   *   The path to check (e.g., /jsonapi/node/{bundle}/{uuid}).
   *
   * @return \Drupal\node\NodeInterface|null
   *   The node entity if found via JSON:API path, or NULL if not a JSON:API
   *   path or node not found.
   */
  private function resolveNodeFromJsonApiPath(string $path): ?NodeInterface {
    // Match JSON:API pattern: /jsonapi/node/{bundle}/{uuid}.
    if (!preg_match('#^/jsonapi/node/([^/]+)/([0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12})#i', $path, $matches)) {
      return NULL;
    }

    $uuid = $matches[2];
    $nodes = $this->entityTypeManager->getStorage('node')->loadByProperties(['uuid' => $uuid]);

    if (empty($nodes)) {
      return NULL;
    }

    /** @var \Drupal\node\NodeInterface $node */
    $node = reset($nodes);
    return $node;
  }

  /**
   * Builds the prompt for summary generation.
   *
   * @param string $body_content
   *   The body content to summarize.
   *
   * @return string
   *   The formatted prompt for the LLM.
   */
  private function buildSummaryPrompt(string $body_content): string {
    $stripped_content = strip_tags($body_content);
    return sprintf(
      "Please provide a concise summary of the following content. The summary should capture the key points in 2-3 sentences:\n\n<text>\n%s\n</text>",
      $stripped_content
    );
  }

}
