<?php

declare(strict_types=1);

namespace Drupal\mcp_server_test\Plugin\tool\Tool;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mcp_server\ClientGatewayAwareInterface;
use Drupal\mcp_server\Traits\McpToolSamplingTrait;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;

/**
 * Test tool that uses ClientGateway for sampling requests.
 *
 * Use this tool with MCP Inspector to manually test sampling functionality.
 */
#[Tool(
  id: 'mcp_server_test:sampling',
  label: new TranslatableMarkup('Sampling Test Tool'),
  description: new TranslatableMarkup('A test tool that requests LLM sampling from the client'),
  operation: ToolOperation::Read,
  input_definitions: [
    'prompt' => new InputDefinition(
      data_type: 'string',
      label: new TranslatableMarkup('Prompt'),
      description: new TranslatableMarkup('The prompt to send for sampling'),
      required: TRUE,
    ),
    'max_tokens' => new InputDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup('Max Tokens'),
      description: new TranslatableMarkup('Maximum tokens to generate (default: 500)'),
      required: FALSE,
    ),
  ],
)]
final class SamplingTestTool extends ToolBase implements ClientGatewayAwareInterface {

  use McpToolSamplingTrait;

  /**
   * {@inheritdoc}
   */
  protected function doExecute(array $values): ExecutableResult {
    $prompt = $values['prompt'] ?? '';
    $max_tokens = $values['max_tokens'] ?? 500;

    try {
      $result = $this->sample($prompt, (int) $max_tokens);

      $content = $result->content;
      $response_text = $content->text ?? '(no text content)';

      return ExecutableResult::success(
        $this->t('Sampling completed successfully'),
        [
          'response' => $response_text,
          'model' => $result->model,
          'stop_reason' => $result->stopReason,
        ]
      );
    }
    catch (\RuntimeException $e) {
      return ExecutableResult::failure($this->t('Sampling failed: @message', ['@message' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(array $values, AccountInterface $account, bool $return_as_object = FALSE): bool|AccessResultInterface {
    $access_result = AccessResult::allowed();
    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

}
