<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\mcp_server\Entity\McpToolConfig;
use Drupal\mcp_server\McpBridgeService;

/**
 * Tests the McpBridgeService integration between Tool API and MCP configs.
 *
 * @group mcp_server
 */
final class McpBridgeServiceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'mcp_server',
    'tool',
    'mcp_server_test',
    'simple_oauth',
    'consumers',
    'serialization',
    'user',
    'system',
    'file',
    'image',
    'options',
  ];

  /**
   * The MCP bridge service.
   */
  private McpBridgeService $mcpBridge;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('consumer');
    $this->installEntitySchema('oauth2_scope');
    $this->installEntitySchema('oauth2_token');
    $this->installConfig(['user', 'simple_oauth', 'tool', 'mcp_server']);
    $this->mcpBridge = $this->container->get('mcp_server.bridge');
  }

  /**
   * Tests complete workflow: enabled tools, disabled filtering, execution.
   */
  public function testCompleteToolManagementWorkflow(): void {
    // Initially no tools should be enabled.
    $enabled_tools = $this->mcpBridge->getEnabledTools();
    $this->assertEmpty($enabled_tools);

    // Create an enabled configuration.
    // Note: The entity ID is used as the MCP tool name (machine-safe).
    $enabled_config = McpToolConfig::create([
      'id' => 'custom_enabled',
      'mcp_tool_name' => 'Custom Enabled Tool',
      'tool_id' => 'mcp_server_test:example',
      'description' => 'Custom description',
      'status' => TRUE,
      'authentication_mode' => 'disabled',
    ]);
    $enabled_config->save();

    // Create a disabled configuration.
    $disabled_config = McpToolConfig::create([
      'id' => 'custom_disabled',
      'mcp_tool_name' => 'Custom Disabled Tool',
      'tool_id' => 'mcp_server_test:example',
      'status' => FALSE,
    ]);
    $disabled_config->save();

    // getEnabledTools() should return only the enabled tool.
    $enabled_tools = $this->mcpBridge->getEnabledTools();
    $this->assertCount(1, $enabled_tools);
    $this->assertArrayHasKey('custom_enabled', $enabled_tools);
    $this->assertArrayNotHasKey('custom_disabled', $enabled_tools);

    // Verify tool structure and overrides.
    $tool = $enabled_tools['custom_enabled'];
    $this->assertEquals('custom_enabled', $tool['id']);
    $this->assertEquals('Custom Enabled Tool', $tool['label']);
    $this->assertEquals('Custom description', $tool['description']);
    $this->assertEquals('mcp_server_test:example', $tool['tool_id']);
    $this->assertArrayHasKey('inputSchema', $tool);

    // Test getMcpTool() for enabled tool.
    $retrieved_tool = $this->mcpBridge->getMcpTool('custom_enabled');
    $this->assertNotNull($retrieved_tool);
    $this->assertEquals('custom_enabled', $retrieved_tool['id']);

    // Test getMcpTool() for disabled tool returns null.
    $disabled_tool = $this->mcpBridge->getMcpTool('custom_disabled');
    $this->assertNull($disabled_tool);

    // Test isToolEnabled().
    $this->assertTrue($this->mcpBridge->isToolEnabled('custom_enabled'));
    $this->assertFalse($this->mcpBridge->isToolEnabled('custom_disabled'));

    // Test getToolApiId().
    $tool_api_id = $this->mcpBridge->getToolApiId('custom_enabled');
    $this->assertEquals('mcp_server_test:example', $tool_api_id);

    // Test execution through MCP bridge.
    $result = $this->mcpBridge->executeMcpTool('custom_enabled', [
      'message' => 'Bridge Test',
    ]);
    $this->assertTrue($result['success']);
    $this->assertEquals('Echo: Bridge Test', $result['data']['result']);
  }

  /**
   * Tests configuration overrides are properly applied.
   */
  public function testConfigurationOverrides(): void {
    // Configuration without description override.
    // Note: The entity ID is used as the MCP tool name (machine-safe).
    $config1 = McpToolConfig::create([
      'id' => 'no_override',
      'mcp_tool_name' => 'No Override Tool',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'disabled',
    ]);
    $config1->save();

    $tool = $this->mcpBridge->getMcpTool('no_override');
    // Should use original tool description.
    $this->assertEquals('A simple test tool that echoes a message with a prefix', $tool['description']);

    // Configuration with description override.
    $config2 = McpToolConfig::create([
      'id' => 'with_override',
      'mcp_tool_name' => 'Overridden Tool',
      'tool_id' => 'mcp_server_test:example',
      'description' => 'Overridden description',
      'status' => TRUE,
      'authentication_mode' => 'disabled',
    ]);
    $config2->save();

    $tool_overridden = $this->mcpBridge->getMcpTool('with_override');
    $this->assertEquals('Overridden description', $tool_overridden['description']);
  }

  /**
   * Tests error handling for invalid tool references.
   */
  public function testInvalidToolReference(): void {
    // Config pointing to non-existent tool.
    $invalid_config = McpToolConfig::create([
      'id' => 'invalid',
      'mcp_tool_name' => 'Invalid',
      'tool_id' => 'nonexistent:tool',
      'status' => TRUE,
    ]);
    $invalid_config->save();

    // Should be filtered out from enabled tools.
    $enabled_tools = $this->mcpBridge->getEnabledTools();
    $this->assertEmpty($enabled_tools);
  }

  /**
   * Tests execution failure handling.
   */
  public function testExecutionFailure(): void {
    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage("MCP tool 'nonexistent' not found or is disabled");
    $this->mcpBridge->executeMcpTool('nonexistent', []);
  }

}
