<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\mcp_server\Entity\McpToolConfig;
use Drupal\Tests\mcp_server\Traits\OAuth2ScopeTestTrait;

/**
 * Tests OAuth scope discovery service.
 *
 * @group mcp_server
 */
final class OAuthScopeDiscoveryServiceTest extends KernelTestBase {

  use OAuth2ScopeTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'file',
    'image',
    'options',
    'mcp_server',
    'tool',
    'simple_oauth',
    'consumers',
    'serialization',
  ];

  /**
   * The scope discovery service.
   *
   * @var \Drupal\mcp_server\Service\OAuthScopeDiscoveryService
   */
  protected $scopeDiscovery;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('mcp_tool_config');
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('consumer');
    $this->installEntitySchema('oauth2_scope');
    $this->installEntitySchema('oauth2_token');
    $this->installConfig(['user', 'simple_oauth', 'mcp_server']);

    $this->scopeDiscovery = $this->container->get('mcp_server.oauth_scope_discovery');
  }

  /**
   * Tests empty scopes when no configs exist.
   */
  public function testEmptyScopes(): void {
    $scopes = $this->scopeDiscovery->getScopesSupported();
    $this->assertSame([], $scopes);
  }

  /**
   * Tests scope aggregation from multiple tools.
   */
  public function testScopeAggregation(): void {
    // Create OAuth2 scope entities.
    $scope_ids = $this->createOauthScopes([
      'content:read',
      'content:write',
      'widget:delete',
    ]);

    // Create first tool config.
    McpToolConfig::create([
      'id' => 'tool1',
      'mcp_tool_name' => 'Tool 1',
      'tool_id' => 'tool1',
      'status' => TRUE,
      'authentication_mode' => 'required',
      'scopes' => [$scope_ids['content:read'], $scope_ids['content:write']],
    ])->save();

    // Create second tool config.
    McpToolConfig::create([
      'id' => 'tool2',
      'mcp_tool_name' => 'Tool 2',
      'tool_id' => 'tool2',
      'status' => TRUE,
      'authentication_mode' => 'required',
      'scopes' => [$scope_ids['widget:delete'], $scope_ids['content:read']],
    ])->save();

    $scopes = $this->scopeDiscovery->getScopesSupported();

    // Should be deduplicated and sorted.
    $expected = ['content:read', 'content:write', 'widget:delete'];
    $this->assertSame($expected, $scopes);
  }

  /**
   * Tests disabled tools are ignored.
   */
  public function testDisabledToolsIgnored(): void {
    // Create OAuth2 scope entities.
    $scope_ids = $this->createOauthScopes([
      'enabled:scope',
      'disabled:scope',
    ]);

    // Enabled tool.
    McpToolConfig::create([
      'id' => 'enabled',
      'mcp_tool_name' => 'Enabled Tool',
      'tool_id' => 'enabled',
      'status' => TRUE,
      'scopes' => [$scope_ids['enabled:scope']],
    ])->save();

    // Disabled tool.
    McpToolConfig::create([
      'id' => 'disabled',
      'mcp_tool_name' => 'Disabled Tool',
      'tool_id' => 'disabled',
      'status' => FALSE,
      'scopes' => [$scope_ids['disabled:scope']],
    ])->save();

    $scopes = $this->scopeDiscovery->getScopesSupported();

    $this->assertContains('enabled:scope', $scopes);
    $this->assertNotContains('disabled:scope', $scopes);
  }

  /**
   * Tests empty scopes array handled correctly.
   */
  public function testEmptyScopesArray(): void {
    McpToolConfig::create([
      'id' => 'no_scopes',
      'mcp_tool_name' => 'No Scopes Tool',
      'tool_id' => 'no_scopes',
      'status' => TRUE,
      'scopes' => [],
    ])->save();

    $scopes = $this->scopeDiscovery->getScopesSupported();
    $this->assertSame([], $scopes);
  }

}
