<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\mcp_server\Entity\McpToolConfig;
use Drupal\mcp_server\McpBridgeService;
use Drupal\mcp_server\Exception\InsufficientScopeException;
use Drupal\mcp_server\Exception\AuthenticationRequiredException;
use Drupal\Tests\mcp_server\Traits\OAuth2ScopeTestTrait;
use Drupal\user\Entity\User;

/**
 * Kernel tests for OAuth scope validation integration.
 *
 * These tests focus on the critical integration paths:
 * - McpToolConfig scope field persistence
 * - Scope enforcement in required authentication mode
 * - Disabled authentication mode (no checks)
 *
 * @group mcp_server
 */
final class OAuthScopeIntegrationTest extends KernelTestBase {

  use OAuth2ScopeTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'mcp_server',
    'tool',
    'mcp_server_test',
    'system',
    'user',
    'file',
    'image',
    'options',
    'simple_oauth',
    'consumers',
    'serialization',
  ];

  /**
   * The MCP bridge service.
   */
  private McpBridgeService $mcpBridge;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('consumer');
    $this->installEntitySchema('oauth2_scope');
    $this->installEntitySchema('oauth2_token');
    $this->installConfig(['user', 'simple_oauth', 'tool', 'mcp_server']);
    $this->mcpBridge = $this->container->get('mcp_server.bridge');
  }

  /**
   * Tests disabled authentication mode allows anonymous access.
   */
  public function testDisabledAuthenticationModeAllowsAnonymous(): void {
    // Create OAuth2 scope entities.
    $scope_ids = $this->createOauthScopes(['admin']);

    // Create tool with disabled authentication.
    // The entity ID is used as the MCP tool name for lookups.
    $config = McpToolConfig::create([
      'id' => 'disabled_auth',
      'mcp_tool_name' => 'Disabled Test',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'disabled',
    ]);
    $config->setScopes([$scope_ids['admin']]);
    $config->save();

    // Execute as anonymous user - should not throw authentication exception.
    // We don't care about tool execution success, only that auth doesn't block.
    try {
      $this->mcpBridge->executeMcpTool('disabled_auth', [
        'message' => 'test message',
      ]);
      // If we get here without exception, test passes.
      $this->addToAssertionCount(1);
    }
    catch (AuthenticationRequiredException | InsufficientScopeException $e) {
      $this->fail('Should not throw auth exception in disabled mode: ' . $e->getMessage());
    }
  }

  /**
   * Tests required authentication mode blocks anonymous users.
   */
  public function testRequiredAuthenticationModeBlocksAnonymous(): void {
    // The entity ID is used as the MCP tool name for lookups.
    $config = McpToolConfig::create([
      'id' => 'required_auth',
      'mcp_tool_name' => 'Required Test',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'required',
    ]);
    $config->save();

    $this->expectException(AuthenticationRequiredException::class);
    $this->mcpBridge->executeMcpTool('required_auth', ['message' => 'test']);
  }

  /**
   * Tests required mode with empty scopes allows any authenticated user.
   */
  public function testRequiredModeEmptyScopesAllowsAuthenticated(): void {
    // Create authenticated user.
    $user = User::create([
      'name' => 'test_user',
      'status' => 1,
    ]);
    $user->save();
    $this->container->get('current_user')->setAccount($user);

    // The entity ID is used as the MCP tool name for lookups.
    $config = McpToolConfig::create([
      'id' => 'empty_scopes',
      'mcp_tool_name' => 'Empty Scopes Test',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'required',
    ]);
    $config->setScopes([]);
    $config->save();

    // Should not throw exception (no scope restrictions for
    // authenticated user).
    try {
      $this->mcpBridge->executeMcpTool('empty_scopes', [
        'message' => 'test message',
      ]);
      // If we get here, auth passed.
      $this->addToAssertionCount(1);
    }
    catch (AuthenticationRequiredException | InsufficientScopeException $e) {
      $this->fail('Should not throw auth exception with empty scopes: ' . $e->getMessage());
    }
  }

  /**
   * Tests scope validation with authenticated user but no token.
   *
   * When user is authenticated but there's no OAuth token (e.g., session auth),
   * required mode with scopes should fail due to missing token scopes.
   */
  public function testRequiredModeWithScopesFailsWithoutToken(): void {
    // Create OAuth2 scope entities.
    $scope_ids = $this->createOauthScopes(['read', 'write']);

    // Create authenticated user.
    $user = User::create([
      'name' => 'test_user',
      'status' => 1,
    ]);
    $user->save();
    $this->container->get('current_user')->setAccount($user);

    // The entity ID is used as the MCP tool name for lookups.
    $config = McpToolConfig::create([
      'id' => 'with_scopes',
      'mcp_tool_name' => 'Scopes Test',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'required',
    ]);
    $config->setScopes([$scope_ids['read'], $scope_ids['write']]);
    $config->save();

    $this->expectException(InsufficientScopeException::class);
    $this->mcpBridge->executeMcpTool('with_scopes', ['message' => 'test']);
  }

  /**
   * Tests that only required and disabled authentication modes are valid.
   */
  public function testAuthenticationModesRestricted(): void {
    $config = McpToolConfig::create([
      'id' => 'mode_test',
      'mcp_tool_name' => 'Mode Test',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
    ]);

    // Valid modes should work.
    $config->setAuthenticationMode('required');
    $this->assertEquals('required', $config->getAuthenticationMode());

    $config->setAuthenticationMode('disabled');
    $this->assertEquals('disabled', $config->getAuthenticationMode());

    // Invalid mode should throw exception.
    $this->expectException(\InvalidArgumentException::class);
    $config->setAuthenticationMode('invalid');
  }

}
