<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * Tests the MCP Resource Template system integration.
 *
 * This test validates the complete resource template system including plugin
 * discovery, entity access control, JSON:API serialization, plugin-based
 * configuration management, and the bridge service integration.
 *
 * @group mcp_server
 */
final class ResourceTemplateSystemTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'jsonapi',
    'serialization',
    'mcp_server',
    'field',
    'text',
    'filter',
    'tool',
    'simple_oauth',
    'consumers',
    'file',
    'image',
    'options',
  ];

  /**
   * The resource template manager.
   *
   * @var \Drupal\mcp_server\Plugin\ResourceTemplateManager
   */
  private $resourceTemplateManager;

  /**
   * The resource bridge service.
   *
   * @var \Drupal\mcp_server\McpResourceBridgeService
   */
  private $resourceBridge;

  /**
   * Test user with view unpublished permission.
   *
   * @var \Drupal\user\Entity\User
   */
  private $editorUser;

  /**
   * Test user without special permissions.
   *
   * @var \Drupal\user\Entity\User
   */
  private $authenticatedUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('file');
    $this->installEntitySchema('consumer');
    $this->installEntitySchema('oauth2_scope');
    $this->installEntitySchema('oauth2_token');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('system', ['sequences']);
    $this->installConfig(['node', 'jsonapi', 'filter', 'system', 'simple_oauth', 'tool']);

    // Create content type.
    NodeType::create([
      'type' => 'page',
      'name' => 'Basic page',
    ])->save();

    // Create anonymous role and grant permission to view published content.
    $anonymous_role = Role::create([
      'id' => 'anonymous',
      'label' => 'Anonymous User',
    ]);
    $anonymous_role->grantPermission('access content');
    $anonymous_role->save();

    // Create test users.
    $this->authenticatedUser = User::create([
      'name' => 'authenticated_user',
      'status' => 1,
    ]);
    $this->authenticatedUser->save();

    // Create editor role with view unpublished permission.
    $editor_role = Role::create([
      'id' => 'editor',
      'label' => 'Editor',
    ]);
    $editor_role->grantPermission('bypass node access');
    $editor_role->save();

    $this->editorUser = User::create([
      'name' => 'editor_user',
      'status' => 1,
    ]);
    $this->editorUser->addRole('editor');
    $this->editorUser->save();

    // Rebuild container to ensure plugin discovery works.
    $this->container->get('kernel')->rebuildContainer();

    // Get services from the rebuilt container.
    // Note: After rebuildContainer(), we need to get a fresh container
    // reference.
    /** @var \Symfony\Component\DependencyInjection\ContainerInterface $container */
    $container = $this->container->get('kernel')->getContainer();
    $this->resourceTemplateManager = $container->get('plugin.manager.mcp_server.resource_template');
    $this->resourceBridge = $container->get('mcp_server.resource_bridge');

    // Clear plugin cache to ensure fresh discovery.
    $this->resourceTemplateManager->clearCachedDefinitions();
  }

  /**
   * Tests plugin discovery and instantiation.
   */
  public function testPluginDiscovery(): void {
    $definitions = $this->resourceTemplateManager->getDefinitions();

    $this->assertArrayHasKey('content_entity', $definitions, 'ContentEntityResourceTemplate plugin is discovered');

    $plugin_def = $definitions['content_entity'];
    $this->assertSame('content_entity', $plugin_def['id']);
    $this->assertArrayHasKey('label', $plugin_def);
    $this->assertArrayHasKey('module_dependencies', $plugin_def);
    $this->assertContains('jsonapi', $plugin_def['module_dependencies']);

    // Instantiate plugin.
    $plugin = $this->resourceTemplateManager->createInstance('content_entity');
    $this->assertInstanceOf(
      'Drupal\mcp_server\Plugin\ResourceTemplateInterface',
      $plugin,
      'Plugin implements ResourceTemplateInterface'
    );
  }

  /**
   * Tests entity type discovery.
   */
  public function testEntityDiscovery(): void {
    $plugin = $this->resourceTemplateManager->createInstance('content_entity');
    $resources = $plugin->getResources();

    $this->assertNotEmpty($resources, 'Plugin discovers content entities');

    // Extract entity types from resources.
    $entity_types = array_map(function ($resource) {
      if (preg_match('#drupal://entity/([^/]+)/#', $resource['uri'], $matches)) {
        return $matches[1];
      }
      return NULL;
    }, $resources);
    $entity_types = array_filter($entity_types);

    $this->assertContains('node', $entity_types, 'Node entity type discovered');

    // Verify URI format.
    $node_resource = array_filter($resources, fn($r) => str_contains($r['uri'], 'drupal://entity/node/'));
    $node_resource = reset($node_resource);
    $this->assertStringContainsString('drupal://entity/node/', $node_resource['uri']);
    $this->assertSame('application/vnd.api+json', $node_resource['mimeType']);
  }

  /**
   * Tests access control for published and unpublished nodes.
   */
  public function testAccessControl(): void {
    // Create published and unpublished nodes.
    $published_node = Node::create([
      'type' => 'page',
      'title' => 'Published Node',
      'status' => 1,
    ]);
    $published_node->save();

    $unpublished_node = Node::create([
      'type' => 'page',
      'title' => 'Unpublished Node',
      'status' => 0,
    ]);
    $unpublished_node->save();

    $plugin = $this->resourceTemplateManager->createInstance('content_entity');

    // Anonymous user can access published.
    $access = $plugin->checkAccess(
      "drupal://entity/node/{$published_node->id()}",
      User::getAnonymousUser()
    );
    $this->assertTrue($access->isAllowed(), 'Anonymous user can access published node');

    // Anonymous user cannot access unpublished.
    $access = $plugin->checkAccess(
      "drupal://entity/node/{$unpublished_node->id()}",
      User::getAnonymousUser()
    );
    $this->assertFalse($access->isAllowed(), 'Anonymous user cannot access unpublished node');

    // Editor can access unpublished.
    $access = $plugin->checkAccess(
      "drupal://entity/node/{$unpublished_node->id()}",
      $this->editorUser
    );
    $this->assertTrue($access->isAllowed(), 'Editor can access unpublished node');

    // Non-existent entity.
    $access = $plugin->checkAccess('drupal://entity/node/99999', User::getAnonymousUser());
    $this->assertFalse($access->isAllowed(), 'Non-existent entity returns forbidden');
  }

  /**
   * Tests JSON:API serialization structure.
   */
  public function testJsonApiSerialization(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test JSON:API Node',
      'status' => 1,
    ]);
    $node->save();

    $plugin = $this->resourceTemplateManager->createInstance('content_entity');
    $content = $plugin->getResourceContent("drupal://entity/node/{$node->id()}");

    $this->assertIsArray($content, 'Resource content is array');
    $this->assertArrayHasKey('data', $content, 'JSON:API response has data key');

    $data = $content['data'];
    $this->assertArrayHasKey('type', $data);
    $this->assertArrayHasKey('id', $data);
    $this->assertArrayHasKey('attributes', $data);
    $this->assertSame('node--page', $data['type']);
    $this->assertArrayHasKey('title', $data['attributes']);
  }

  /**
   * Tests plugin configuration CRUD operations.
   */
  public function testPluginConfiguration(): void {
    $config = $this->container->get('config.factory')->getEditable('mcp_server.resource_plugins');

    // Create plugin configuration.
    $plugins = [
      [
        'id' => 'content_entity',
        'enabled' => TRUE,
        'configuration' => [],
      ],
    ];
    $config->set('plugins', $plugins)->save();

    // Load config.
    $loaded = $this->container->get('config.factory')->get('mcp_server.resource_plugins');
    $loaded_plugins = $loaded->get('plugins');
    $this->assertNotEmpty($loaded_plugins, 'Plugin configuration saved and loaded');
    $this->assertSame('content_entity', $loaded_plugins[0]['id']);
    $this->assertTrue($loaded_plugins[0]['enabled']);

    // Update config to disable plugin.
    $loaded_plugins[0]['enabled'] = FALSE;
    $config->set('plugins', $loaded_plugins)->save();
    $reloaded = $this->container->get('config.factory')->get('mcp_server.resource_plugins');
    $reloaded_plugins = $reloaded->get('plugins');
    $this->assertFalse($reloaded_plugins[0]['enabled']);

    // Clear config.
    $config->set('plugins', [])->save();
    $cleared = $this->container->get('config.factory')->get('mcp_server.resource_plugins');
    $this->assertEmpty($cleared->get('plugins'), 'Plugin configuration cleared');
  }

  /**
   * Tests bridge service integration with plugin configurations.
   */
  public function testBridgeServiceIntegration(): void {
    $config = $this->container->get('config.factory')->getEditable('mcp_server.resource_plugins');

    // Initially no resources should be enabled.
    $this->container->get('cache.default')->deleteAll();
    $resources = $this->resourceBridge->getEnabledResources();
    $this->assertEmpty($resources, 'Initially no resources enabled');

    // Enable content_entity plugin.
    $plugins = [
      [
        'id' => 'content_entity',
        'enabled' => TRUE,
        'configuration' => [],
      ],
    ];
    $config->set('plugins', $plugins)->save();

    // Clear cache to force refresh.
    $this->container->get('cache.default')->deleteAll();

    // Get enabled resources.
    $resources = $this->resourceBridge->getEnabledResources();
    $this->assertNotEmpty($resources, 'Bridge service returns enabled resources');

    // Create test node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Bridge Test Node',
      'status' => 1,
    ]);
    $node->save();

    // Get resource content via bridge.
    $content = $this->resourceBridge->getResourceContent("drupal://entity/node/{$node->id()}");
    $this->assertIsArray($content, 'Bridge service returns content');
    $this->assertArrayHasKey('data', $content);

    // Check access via bridge.
    $access = $this->resourceBridge->checkResourceAccess(
      "drupal://entity/node/{$node->id()}",
      User::getAnonymousUser()
    );
    $this->assertTrue($access->isAllowed(), 'Bridge service checks access');

    // Disable plugin.
    $plugins[0]['enabled'] = FALSE;
    $config->set('plugins', $plugins)->save();
    $this->container->get('cache.default')->deleteAll();

    $resources = $this->resourceBridge->getEnabledResources();
    $this->assertEmpty($resources, 'Disabled resources not returned');

    // Clear config.
    $config->set('plugins', [])->save();
  }

  /**
   * Tests complete end-to-end workflow.
   *
   * This test validates the full flow from configuration creation through
   * content retrieval and access control enforcement.
   */
  public function testEndToEndWorkflow(): void {
    // 1. Create plugin configuration.
    $config = $this->container->get('config.factory')->getEditable('mcp_server.resource_plugins');
    $plugins = [
      [
        'id' => 'content_entity',
        'enabled' => TRUE,
        'configuration' => [],
      ],
    ];
    $config->set('plugins', $plugins)->save();

    // 2. Create test content.
    $node = Node::create([
      'type' => 'page',
      'title' => 'E2E Test Node',
      'status' => 1,
    ]);
    $node->save();

    // 3. Clear cache.
    $this->container->get('cache.default')->deleteAll();

    // 4. Get enabled resources via bridge.
    $resources = $this->resourceBridge->getEnabledResources();
    $this->assertNotEmpty($resources, 'E2E: Resources enabled');

    // 5. Access resource content.
    $content = $this->resourceBridge->getResourceContent("drupal://entity/node/{$node->id()}");
    $this->assertIsArray($content, 'E2E: Content retrieved');
    $this->assertArrayHasKey('data', $content);
    $this->assertSame('E2E Test Node', $content['data']['attributes']['title']);

    // 6. Verify access control.
    $access = $this->resourceBridge->checkResourceAccess(
      "drupal://entity/node/{$node->id()}",
      User::getAnonymousUser()
    );
    $this->assertTrue($access->isAllowed(), 'E2E: Access control enforced');

    // Cleanup.
    $config->set('plugins', [])->save();
  }

  /**
   * Tests invalid URI handling.
   */
  public function testInvalidUriHandling(): void {
    $plugin = $this->resourceTemplateManager->createInstance('content_entity');

    // Test invalid URI format.
    $content = $plugin->getResourceContent('invalid://uri/format');
    $this->assertNull($content, 'Invalid URI returns null content');

    $access = $plugin->checkAccess('invalid://uri/format', User::getAnonymousUser());
    $this->assertFalse($access->isAllowed(), 'Invalid URI returns forbidden access');

    // Test non-existent entity type.
    $content = $plugin->getResourceContent('drupal://entity/nonexistent/123');
    $this->assertNull($content, 'Non-existent entity type returns null content');

    $access = $plugin->checkAccess('drupal://entity/nonexistent/123', User::getAnonymousUser());
    $this->assertFalse($access->isAllowed(), 'Non-existent entity type returns forbidden access');
  }

  /**
   * Tests plugin configuration validation.
   */
  public function testConfigurationValidation(): void {
    $config = $this->container->get('config.factory')->getEditable('mcp_server.resource_plugins');

    // Test plugin configuration structure.
    $plugins = [
      [
        'id' => 'content_entity',
        'enabled' => TRUE,
        'configuration' => [],
      ],
    ];
    $config->set('plugins', $plugins)->save();

    $loaded = $this->container->get('config.factory')->get('mcp_server.resource_plugins');
    $loaded_plugins = $loaded->get('plugins');
    $this->assertIsArray($loaded_plugins[0]['configuration'], 'Configuration is array');
    $this->assertIsBool($loaded_plugins[0]['enabled'], 'Enabled is boolean');
    $this->assertIsString($loaded_plugins[0]['id'], 'ID is string');
  }

  /**
   * Tests invalid plugin reference handling.
   */
  public function testInvalidPluginReference(): void {
    $config = $this->container->get('config.factory')->getEditable('mcp_server.resource_plugins');

    // Create config with non-existent plugin.
    $plugins = [
      [
        'id' => 'nonexistent_plugin',
        'enabled' => TRUE,
        'configuration' => [],
      ],
    ];
    $config->set('plugins', $plugins)->save();

    // Clear cache.
    $this->container->get('cache.default')->deleteAll();

    // Should not return resources from invalid plugin.
    $resources = $this->resourceBridge->getEnabledResources();
    $this->assertEmpty($resources, 'Invalid plugin reference does not return resources');

    // Cleanup.
    $config->set('plugins', [])->save();
  }

}
