<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\mcp_server\ToolApiDiscovery;

/**
 * Tests the ToolApiDiscovery service integration with Tool API.
 *
 * @group mcp_server
 */
final class ToolApiDiscoveryTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'mcp_server',
    'tool',
    'mcp_server_test',
    'simple_oauth',
    'consumers',
    'serialization',
    'user',
    'system',
    'file',
    'image',
    'options',
  ];

  /**
   * The ToolApiDiscovery service.
   */
  private ToolApiDiscovery $toolApiDiscovery;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('consumer');
    $this->installEntitySchema('oauth2_scope');
    $this->installEntitySchema('oauth2_token');
    $this->installConfig(['user', 'simple_oauth', 'tool', 'mcp_server']);
    $this->toolApiDiscovery = $this->container->get('mcp_server.tool_api_discovery');
  }

  /**
   * Tests getting a specific tool definition with schema conversion.
   */
  public function testGetToolDefinition(): void {
    $definition = $this->toolApiDiscovery->getToolDefinition('mcp_server_test:example');

    // Should return an array with complete definition.
    $this->assertIsArray($definition);
    $this->assertArrayHasKey('id', $definition);
    $this->assertArrayHasKey('label', $definition);
    $this->assertArrayHasKey('description', $definition);
    $this->assertArrayHasKey('inputSchema', $definition);
    $this->assertArrayHasKey('destructive', $definition);

    // Verify schema structure.
    $schema = $definition['inputSchema'];
    $this->assertArrayHasKey('type', $schema);
    $this->assertEquals('object', $schema['type']);
    $this->assertArrayHasKey('properties', $schema);

    // Verify input definitions are converted.
    $properties = $schema['properties'];
    $this->assertArrayHasKey('message', $properties);
    $this->assertArrayHasKey('prefix', $properties);
    $this->assertArrayHasKey('count', $properties);

    // Verify required field is marked.
    $this->assertArrayHasKey('required', $schema);
    $this->assertContains('message', $schema['required']);
    $this->assertNotContains('prefix', $schema['required']);

    // Verify type conversions.
    $this->assertEquals('string', $properties['message']['type']);
    $this->assertEquals('integer', $properties['count']['type']);
  }

  /**
   * Tests getToolDefinition with invalid tool ID.
   */
  public function testGetToolDefinitionInvalidId(): void {
    $definition = $this->toolApiDiscovery->getToolDefinition('invalid:tool');
    $this->assertNull($definition);
  }

  /**
   * Tests tool execution success path.
   */
  public function testExecuteToolSuccess(): void {
    $result = $this->toolApiDiscovery->executeTool('mcp_server_test:example', [
      'message' => 'Hello World',
      'prefix' => 'Test',
      'count' => 2,
    ]);

    // Verify result structure.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('success', $result);
    $this->assertArrayHasKey('message', $result);
    $this->assertArrayHasKey('data', $result);

    // Verify execution success.
    $this->assertTrue($result['success']);
    $this->assertStringContainsString('Successfully processed message', (string) $result['message']);
    $this->assertIsArray($result['data']);
    $this->assertEquals('Test: Hello World | Test: Hello World', $result['data']['result']);
  }

  /**
   * Tests tool execution failure with invalid tool ID.
   */
  public function testExecuteToolInvalidId(): void {
    $this->expectException(\RuntimeException::class);
    $this->toolApiDiscovery->executeTool('invalid:tool', []);
  }

}
