<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\mcp_server\Entity\McpToolConfig;
use Drupal\mcp_server\McpBridgeService;
use Drupal\Tests\mcp_server\Traits\OAuth2ScopeTestTrait;

/**
 * Tests tool authorization metadata in tools list.
 *
 * @group mcp_server
 */
final class ToolAuthorizationMetadataTest extends KernelTestBase {

  use OAuth2ScopeTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'mcp_server',
    'tool',
    'mcp_server_test',
    'simple_oauth',
    'consumers',
    'serialization',
    'user',
    'system',
    'file',
    'image',
    'options',
  ];

  /**
   * The MCP bridge service.
   */
  private McpBridgeService $mcpBridge;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('consumer');
    $this->installEntitySchema('oauth2_scope');
    $this->installEntitySchema('oauth2_token');
    $this->installConfig(['user', 'simple_oauth', 'tool', 'mcp_server']);
    $this->mcpBridge = $this->container->get('mcp_server.bridge');
  }

  /**
   * Tests authorization field for required auth mode.
   */
  public function testRequiredAuthorization(): void {
    // Create OAuth2 scope entities.
    $scope_ids = $this->createOauthScopes([
      'test:read',
      'test:write',
    ]);

    // Create tool config with required auth.
    $config = McpToolConfig::create([
      'id' => 'required_tool',
      'mcp_tool_name' => 'required_tool',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'required',
    ]);
    $config->setScopes([$scope_ids['test:read'], $scope_ids['test:write']]);
    $config->save();

    // Get enabled tools.
    $tools = $this->mcpBridge->getEnabledTools();

    // Find our tool.
    $this->assertArrayHasKey('required_tool', $tools);
    $tool = $tools['required_tool'];

    // Verify authorization field exists.
    $this->assertArrayHasKey('authorization', $tool);
    $this->assertSame('oauth2', $tool['authorization']['method']);
    $this->assertSame('required', $tool['authorization']['mode']);
    $this->assertContains('test:read', $tool['authorization']['requiredScopes']);
    $this->assertContains('test:write', $tool['authorization']['requiredScopes']);
  }

  /**
   * Tests no authorization field for disabled auth mode.
   */
  public function testDisabledAuthorization(): void {
    // Create tool config with disabled auth.
    McpToolConfig::create([
      'id' => 'disabled_tool',
      'mcp_tool_name' => 'disabled_tool',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'disabled',
    ])->save();

    // Get enabled tools.
    $tools = $this->mcpBridge->getEnabledTools();

    // Find our tool.
    $this->assertArrayHasKey('disabled_tool', $tools);
    $tool = $tools['disabled_tool'];

    // Authorization field should not exist.
    $this->assertArrayNotHasKey('authorization', $tool);
  }

  /**
   * Tests empty scopes array in authorization.
   */
  public function testEmptyScopes(): void {
    // Create tool config with required auth but no scopes.
    $config = McpToolConfig::create([
      'id' => 'no_scopes_tool',
      'mcp_tool_name' => 'no_scopes_tool',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'required',
    ]);
    $config->setScopes([]);
    $config->save();

    // Get enabled tools.
    $tools = $this->mcpBridge->getEnabledTools();

    // Find our tool.
    $this->assertArrayHasKey('no_scopes_tool', $tools);
    $tool = $tools['no_scopes_tool'];

    // Verify authorization field exists with empty scopes.
    $this->assertArrayHasKey('authorization', $tool);
    $this->assertSame('oauth2', $tool['authorization']['method']);
    $this->assertSame('required', $tool['authorization']['mode']);
    $this->assertIsArray($tool['authorization']['requiredScopes']);
    $this->assertEmpty($tool['authorization']['requiredScopes']);
  }

  /**
   * Tests multiple tools with different auth modes.
   */
  public function testMultipleToolsWithDifferentAuthModes(): void {
    // Create OAuth2 scope entities.
    $scope_ids = $this->createOauthScopes(['scope1']);

    // Create tools with different auth modes.
    $config1 = McpToolConfig::create([
      'id' => 'tool1',
      'mcp_tool_name' => 'tool1',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'required',
    ]);
    $config1->setScopes([$scope_ids['scope1']]);
    $config1->save();

    McpToolConfig::create([
      'id' => 'tool2',
      'mcp_tool_name' => 'tool2',
      'tool_id' => 'mcp_server_test:example',
      'status' => TRUE,
      'authentication_mode' => 'disabled',
    ])->save();

    // Get enabled tools.
    $tools = $this->mcpBridge->getEnabledTools();

    // Verify both tools are present.
    $this->assertCount(2, $tools);

    // Verify tool1 has required auth.
    $this->assertArrayHasKey('authorization', $tools['tool1']);
    $this->assertSame('required', $tools['tool1']['authorization']['mode']);

    // Verify tool2 has no auth field.
    $this->assertArrayNotHasKey('authorization', $tools['tool2']);
  }

}
