<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Unit\Session;

use Drupal\mcp_server\Session\SessionContext;
use Drupal\Tests\UnitTestCase;

/**
 * Tests SessionContext value object business logic.
 *
 * @group mcp_server
 * @coversDefaultClass \Drupal\mcp_server\Session\SessionContext
 */
final class SessionContextTest extends UnitTestCase {

  /**
   * Tests expiry checking logic.
   *
   * @covers ::isExpired
   */
  public function testExpiryChecking(): void {
    $futureContext = new SessionContext(
      sessionId: 'test-session-id',
      expiry: time() + 3600,
      roots: [],
    );

    $this->assertFalse($futureContext->isExpired());

    $pastContext = new SessionContext(
      sessionId: 'test-session-id-2',
      expiry: time() - 3600,
      roots: [],
    );

    $this->assertTrue($pastContext->isExpired());
  }

  /**
   * Tests filesystem root validation logic.
   *
   * @covers ::isRootAllowed
   */
  public function testRootValidation(): void {
    $context = new SessionContext(
      sessionId: 'test-session-id',
      expiry: time() + 3600,
      roots: ['/allowed/path', '/another/path'],
    );

    $this->assertTrue($context->isRootAllowed('/allowed/path'));
    $this->assertTrue($context->isRootAllowed('/another/path'));
    $this->assertFalse($context->isRootAllowed('/forbidden/path'));
  }

  /**
   * Tests session context construction with all parameters.
   *
   * @covers ::__construct
   */
  public function testConstruction(): void {
    $expiry = time() + 7200;
    $roots = ['/path/one', '/path/two'];

    $context = new SessionContext(
      sessionId: 'session-abc-123',
      expiry: $expiry,
      roots: $roots,
    );

    $this->assertEquals('session-abc-123', $context->sessionId);
    $this->assertEquals($expiry, $context->expiry);
    $this->assertEquals($roots, $context->roots);
  }

  /**
   * Tests empty roots array is allowed.
   *
   * @covers ::isRootAllowed
   */
  public function testEmptyRoots(): void {
    $context = new SessionContext(
      sessionId: 'test-session-id',
      expiry: time() + 3600,
      roots: [],
    );

    $this->assertFalse($context->isRootAllowed('/any/path'));
  }

}
