<?php

namespace Drupal\mdsitemap\Controller;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;

/**
 * Generates the MD Sitemap for LLM crawlers.
 */
final class SitemapController extends ControllerBase {

  /**
   * The cache backend for mdsitemap.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cacheBackend;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a SitemapController object.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   The cache backend for mdsitemap.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   *
   * @phpstan-param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   */
  public function __construct(
    CacheBackendInterface $cache_backend,
    RequestStack $request_stack,
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->cacheBackend = $cache_backend;
    $this->requestStack = $request_stack;
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    /** @var \Drupal\Core\Cache\CacheBackendInterface $cache */
    $cache = $container->get('cache.mdsitemap');

    return new static(
      $cache,
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
    );
  }

  /**
   * Builds the MD Sitemap response for LLM crawlers.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response containing the sitemap.
   */
  public function build(): Response {
    $urls = [];
    $cache = $this->cacheBackend->get('mdsitemap_sitemap');
    $currentRequest = $this->requestStack->getCurrentRequest();
    $host = '';

    if ($currentRequest instanceof Request) {
      $host = $currentRequest->getSchemeAndHttpHost();
    }

    if ($cache) {
      $urls = is_array($cache->data) ? $cache->data : [];
    }
    else {
      $config = $this->configFactory->get('mdsitemap.settings');
      $suffix = $config->get('url_suffix');

      if (!is_string($suffix) || $suffix === '') {
        $suffix = '.md';
      }

      $entities = $config->get('entities');

      if (!is_array($entities)) {
        $entities = [];
      }

      foreach ($entities as $entity_type_id => $bundles) {
        if (!is_string($entity_type_id) || empty($bundles)) {
          continue;
        }

        $storage = $this->entityTypeManager->getStorage($entity_type_id);
        $definition = $this->entityTypeManager->getDefinition($entity_type_id);

        $query = $storage->getQuery();

        if ($definition->hasKey('status')) {
          $query->condition('status', 1);
        }

        $query->accessCheck(TRUE);
        $ids = $query->execute();

        foreach ($ids as $id) {
          $entity = $storage->load($id);

          if ($entity && is_array($bundles) && in_array($entity->bundle(), $bundles, TRUE) && $entity->hasLinkTemplate('canonical')) {
            $title = $entity->label();
            $entity_url = $entity->toUrl();
            $entity_url_string = $entity_url->toString();
            $concat_url = $host . $entity_url_string . $suffix;

            $urls[] = sprintf(
              "-\n  Title: %s\n  URL: %s",
              $title,
              $concat_url
            );
          }
        }
      }

      $this->cacheBackend->set('mdsitemap_sitemap', $urls, Cache::PERMANENT, ['mdsitemap']);
    }

    $response = new Response(implode("\n", $urls));
    $response->headers->set('Content-Type', 'text/markdown');

    return $response;
  }

}
