<?php

namespace Drupal\Tests\mdsitemap\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\mdsitemap\Controller\SitemapController;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;

/**
 * Tests that sitemap endpoint returns expected URL for a published node.
 *
 * @group mdsitemap
 */
class MdsitemapSitemapTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   *
   * @var string[]
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'mdsitemap',
  ];

  /**
   * Sets up the environment for each test.
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);

    // Create a minimal 'page' content type.
    NodeType::create([
      'type' => 'page',
      'name' => 'Page',
    ])->save();

    $this->installConfig(['mdsitemap']);

    \Drupal::service('config.typed')->clearCachedDefinitions();

    $this->config('mdsitemap.settings')
      ->set('entities', [
        'node' => ['page'],
      ])
      ->set('url_suffix', '.md')
      ->save();

    // Grant anonymous users access to content so accessCheck(TRUE) passes.
    $role = Role::load(RoleInterface::ANONYMOUS_ID);

    if (!$role) {
      // Create the anonymous role if it doesn't exist yet.
      $role = Role::create([
        'id' => RoleInterface::ANONYMOUS_ID,
        'label' => 'Anonymous user',
      ]);
    }

    $role->grantPermission('access content');
    $role->save();
  }

  /**
   * Tests that the sitemap endpoint outputs the expected node URL and title.
   */
  public function testSitemapOutputsNode(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Page',
      'status' => 1,
    ]);
    $node->save();

    $this->assertTrue($node->isPublished());

    /** @var \Drupal\mdsitemap\Controller\SitemapController $controller */
    $controller = $this->container->get('class_resolver')
      ->getInstanceFromDefinition(SitemapController::class);

    $response = $controller->build();
    $content = (string) $response->getContent();

    $this->assertStringContainsString('/node/' . $node->id() . '.md', $content);
    $this->assertStringContainsString('Test Page', $content);
  }

}
