<?php

namespace Drupal\mdsitemap\Form;

use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Provides a settings form for the MDSitemap module.
 */
final class SitemapSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $bundleInfo;

  /**
   * The cache backend for mdsitemap.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cacheBackend;

  public function __construct(EntityTypeManagerInterface $entityTypeManager, EntityTypeBundleInfoInterface $bundleInfo, CacheBackendInterface $cacheBackend) {
    $this->entityTypeManager = $entityTypeManager;
    $this->bundleInfo = $bundleInfo;
    $this->cacheBackend = $cacheBackend;
  }

  /**
   * {@inheritdoc}
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   Returns an instance of this form.
   */
  public static function create(ContainerInterface $container): static {
    /** @var \Drupal\Core\Cache\CacheBackendInterface $cache */
    $cache = $container->get('cache.mdsitemap');

    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $cache,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'mdsitemap_settings_form';
  }

  /**
   * {@inheritdoc}
   *
   * @return array<int,string>
   *   An array of configuration object names that are editable.
   */
  protected function getEditableConfigNames(): array {
    return ['mdsitemap.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string,mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array<string,mixed>
   *   The form structure.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('mdsitemap.settings');
    $entities_config = $config->get('entities') ?? [];

    $form['url_suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL suffix'),
      '#description' => $this->t('Suffix to append to sitemap URLs (e.g. ".md" or ".txt").'),
      '#default_value' => $config->get('url_suffix') ?? '.md',
      '#required' => TRUE,
    ];

    $form['entities'] = [
      '#type' => 'details',
      '#title' => $this->t('Entity inclusion'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $definition) {
      if ($definition->hasLinkTemplate('canonical')) {
        $bundles = $this->bundleInfo->getBundleInfo($entity_type_id);
        $options = [];

        foreach ($bundles as $bundle_id => $bundle) {
          $options[$bundle_id] = $bundle['label'];
        }

        $form['entities'][$entity_type_id] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('@label bundles', ['@label' => $definition->getLabel()]),
          '#options' => $options,
          '#default_value' => (is_array($entities_config) && isset($entities_config[$entity_type_id])) ? $entities_config[$entity_type_id] : [],
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string,mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @param-out array<mixed,mixed> $form
   *   The form structure after processing.
   *
   * @return void
   *   Does not return anything.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->configFactory()->getEditable('mdsitemap.settings');
    $config->set('url_suffix', $form_state->getValue('url_suffix'));
    $entities = [];
    $entities_input = $form_state->getValue('entities');

    if (is_array($entities_input)) {
      foreach ($entities_input as $entity_type => $bundles) {
        if (is_array($bundles)) {
          $filtered = array_keys(array_filter($bundles));
          $entities[$entity_type] = $filtered;
        }
      }
    }

    $config->set('entities', $entities);
    $config->save();

    // Invalidate cache when settings change.
    $this->cacheBackend->deleteAll();

    parent::submitForm($form, $form_state);
  }

}
