<?php

namespace Drupal\mdsitemap;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Generates and caches the Markdown sitemap.
 */
class MdsitemapGenerator {

  public function __construct(
    protected CacheBackendInterface $cacheBackend,
    protected RequestStack $requestStack,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Generates (or returns cached) sitemap content.
   *
   * @return string
   *   The sitemap content in Markdown format.
   */
  public function generate(): string {
    // Check cache first.
    if ($cache = $this->cacheBackend->get('mdsitemap_sitemap')) {
      if (is_array($cache->data)) {
        return implode("\n", $cache->data);
      }
      if (is_string($cache->data)) {
        return $cache->data;
      }
      return '';
    }

    $urls = [];
    $currentRequest = $this->requestStack->getCurrentRequest();
    $host = $currentRequest instanceof Request ? $currentRequest->getSchemeAndHttpHost() : '';

    $config = $this->configFactory->get('mdsitemap.settings');
    $suffix = $config->get('url_suffix');

    if (!is_string($suffix) || $suffix === '') {
      $suffix = '.md';
    }

    $entities = $config->get('entities') ?: [];

    if (!is_array($entities)) {
      $entities = [];
    }

    foreach ($entities as $entity_type_id => $bundles) {
      if (!is_string($entity_type_id) || !is_array($bundles) || $bundles === []) {
        continue;
      }

      $storage = $this->entityTypeManager->getStorage($entity_type_id);
      $definition = $this->entityTypeManager->getDefinition($entity_type_id);

      $query = $storage->getQuery();

      if ($definition->hasKey('status')) {
        $query->condition('status', 1);
      }

      $query->accessCheck(TRUE);

      foreach ($query->execute() as $id) {
        $entity = $storage->load($id);

        if ($entity && in_array($entity->bundle(), $bundles, TRUE) && $entity->hasLinkTemplate('canonical')) {
          $urls[] = sprintf(
            "-\n  Title: %s\n  URL: %s%s",
            (string) $entity->label(),
            $host . $entity->toUrl()->toString(),
            (string) $suffix
          );
        }
      }
    }

    // Cache permanently with a tag so we can invalidate on content changes.
    $this->cacheBackend->set('mdsitemap_sitemap', $urls, Cache::PERMANENT, ['mdsitemap']);

    return implode("\n", $urls);
  }

}
