<?php

declare(strict_types=1);

namespace Drupal\me_language_visibility\Plugin\paragraphs\Behavior;

use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\language\ConfigurableLanguageManagerInterface;
use Drupal\paragraphs\Attribute\ParagraphsBehavior;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\ParagraphInterface;
use Drupal\paragraphs\ParagraphsBehaviorBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a way to define grid based layouts.
 *
 * @ParagraphsBehavior(
 *   id = "me_language_visibility",
 *   label = @Translation("Mercury Editor Language Visibility"),
 *   description = @Translation("Provides paragraphs that are configurable for visibility by language."),
 *   weight = 0
 * )
 */
class LanguageVisibilityBehavior extends ParagraphsBehaviorBase {

  protected RouteMatchInterface $routeMatch;

  protected LanguageManagerInterface $languageManager;

  /**
   * @inheritDoc
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->routeMatch = $container->get('current_route_match');
    $instance->languageManager = $container->get('language_manager');
    return $instance;
  }

  public function buildBehaviorForm(ParagraphInterface $paragraph, array &$form, FormStateInterface $form_state) {
    $options = [];
    foreach ($this->languageManager->getLanguages() as $langcode => $language) {
      $options[$langcode] = $language->isLocked() ? t('- @name -', ['@name' => $language->getName()]) : $language->getName();
    }
    $values = $paragraph->getBehaviorSetting('me_language_visibility', 'languages') ?? array_keys($options);
    $form['details'] = [
        '#type' => 'details',
        '#title' => $this->t('Language visibility'),
    ];
    $form['details']['languages'] = [
      '#type' => 'checkboxes',
      '#options' => $options,
      '#title' => $this->t('Select languages for display'),
      '#default_value' => $values,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitBehaviorForm(ParagraphInterface $paragraph, array &$form, FormStateInterface $form_state) {
    $filtered_values = $this->filterBehaviorFormSubmitValues($paragraph, $form, $form_state);
    // Shift off the wrapper key.
    $filtered_values = $filtered_values['details'] ?? [];
    $paragraph->setBehaviorSettings($this->getPluginId(), $filtered_values);
  }

  /**
   * @inheritDoc
   */
  public function view(array &$build, Paragraph $paragraph, EntityViewDisplayInterface $display, $view_mode): void {
    // If we are in mercury editor preview, there is no work to do.
    // If languages are not configurable, there is no work to do.
    if (
      str_contains($this->routeMatch->getRouteName(), 'mercury_editor')
      || !($this->languageManager instanceof ConfigurableLanguageManagerInterface)
    ) {
      return;
    }
    $language = $this->languageManager->getCurrentLanguage();
    $allowedLanguages = $paragraph->getBehaviorSetting('me_language_visibility', 'languages');
    if (empty($allowedLanguages) || in_array($language->getId(), $allowedLanguages, true)) {
      // Display the build as it is.
      return;
    }
    // Language is not allowed - do not render.
    $build = [];
  }

}
