/**
 * @file
 * JS for ajax and select all features.
 */
(function (Drupal, once) {
  const selectAll = document.getElementById('select-all-media');
  const bulkApplyBtn = document.getElementById('bulk-apply-btn');

  function getSelectedIds() {
    return Array.from(document.querySelectorAll('.media-select:checked')).map(
      (cb) => cb.dataset.mediaId,
    );
  }

  function updateButtonState() {
    const selected = getSelectedIds();
    bulkApplyBtn.disabled = selected.length === 0;
  }

  if (selectAll) {
    selectAll.addEventListener('change', function () {
      document.querySelectorAll('.media-select').forEach((cb) => {
        cb.checked = selectAll.checked;
      });
      updateButtonState();
    });
  }

  document.querySelectorAll('.media-select').forEach((cb) => {
    cb.addEventListener('change', updateButtonState);
  });

  if (bulkApplyBtn) {
    bulkApplyBtn.addEventListener('click', function () {
      const selected = getSelectedIds();
      if (!selected.length) {
        return;
      }

      const url = `${Drupal.url('admin/content/media-accessibility-audit/bulk-apply')}?ids=${selected.join(',')}`;
      window.location.href = url;
    });
  }

  document.addEventListener('click', function (e) {
    const link = e.target.closest('.refresh-suggestion-ajax');
    if (!link) return;

    e.preventDefault();

    fetch(link.href, {
      headers: { Accept: 'application/json' },
    })
      .then((res) => res.json())
      .then((data) => {
        const target = document.getElementById(data.target);
        if (target) {
          target.outerHTML = data.html;
        }
      });
  });

  Drupal.behaviors.mediaAccessibilityRefresh = {
    attach(context) {
      const thinkingPhrases = [
        "AI is thinking…",
        "Reviewing existing alt text…",
        "Checking image context…",
        "Looking for accessibility improvements…",
        "Almost there…",
      ];

      function randomPhrase() {
        return thinkingPhrases[
          Math.floor(Math.random() * thinkingPhrases.length)
        ];
      }

      // Event delegation — survives AJAX DOM replacement
      document.addEventListener('click', function (e) {
        const link = e.target.closest('.refresh-suggestion-ajax');
        if (!link) return;

        const wrapper = link.closest('.suggestion-wrapper');
        if (!wrapper) return;

        const loader = wrapper.querySelector('.suggestion-loader');
        const thinking = wrapper.querySelector('.suggestion-thinking');

        /* Enter loading state */
        link.classList.add('is-loading');
        link.setAttribute('aria-busy', 'true');
        link.style.pointerEvents = 'none';
        link.style.opacity = '0.6';

        if (loader) {
          loader.style.display = 'inline-block';
        }

        if (thinking) {
          thinking.textContent = randomPhrase();
          thinking.style.display = 'inline';
        }
      });

      /* Cleanup after AJAX success */
      document.addEventListener('ajaxComplete', function (event) {
        const wrapper = event.target.closest?.('.suggestion-wrapper');
        if (!wrapper) return;

        const link = wrapper.querySelector('.refresh-suggestion-ajax');
        const loader = wrapper.querySelector('.suggestion-loader');
        const thinking = wrapper.querySelector('.suggestion-thinking');

        if (link) {
          link.classList.remove('is-loading');
          link.removeAttribute('aria-busy');
          link.style.pointerEvents = '';
          link.style.opacity = '';
        }

        if (loader) {
          loader.style.display = 'none';
        }

        if (thinking) {
          thinking.style.display = 'none';
        }
      });

      /* Safety net */
      document.addEventListener('ajaxError', function () {
        document
          .querySelectorAll('.refresh-suggestion-ajax.is-loading')
          .forEach((link) => {
            link.classList.remove('is-loading');
            link.removeAttribute('aria-busy');
            link.style.pointerEvents = '';
            link.style.opacity = '';
          });

        document
          .querySelectorAll('.suggestion-loader, .suggestion-thinking')
          .forEach((el) => {
            el.style.display = 'none';
          });
      });
    },
  };
})(Drupal, once);
