<?php

declare(strict_types=1);

namespace Drupal\media_accessibility_audit\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\media\MediaInterface;
use Drupal\media_accessibility_audit\Service\MediaAccessibilityAuditor;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for Media Accessibility Audit.
 */
final class MediaAccessibilityAuditCommands extends DrushCommands {

  public function __construct(
    protected MediaAccessibilityAuditor $auditor,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected MessengerInterface $messenger,
  ) {
    parent::__construct();
  }

  /**
   * Audit media images for accessibility issues.
   *
   * @command media_accessibility_audit:audit
   * @aliases maa-audit
   * @option batch Run the audit in batch mode for large sites.
   */
  public function audit(array $options = ['batch' => FALSE]): void {
    $this->output()->writeln('Starting Media Accessibility Audit…');

    $media_ids = $this->entityTypeManager
      ->getStorage('media')
      ->getQuery()
      ->accessCheck(FALSE)
      ->execute();

    if (empty($media_ids)) {
      $this->output()->writeln('No media items found.');
      return;
    }

    if (!empty($options['batch'])) {
      $this->runBatch($media_ids);
      return;
    }

    // Non-batch (CLI-friendly).
    $this->runInlineAudit($media_ids);
  }

  /**
   * Runs audit immediately (non-batch).
   */
  protected function runInlineAudit(array $media_ids): void {
    $storage = $this->entityTypeManager->getStorage('media');
    $total_issues = 0;

    foreach ($storage->loadMultiple($media_ids) as $media) {
      if (!$media instanceof MediaInterface) {
        continue;
      }

      $issues = $this->auditor->audit($media);
      if ($issues) {
        $total_issues += count($issues);
        foreach ($issues as $issue) {
          $this->output()->writeln(
            sprintf('⚠️ Media ID %d: %s', $media->id(), $issue)
          );
        }
      }
    }

    $this->output()->writeln("Audit complete. Issues found: {$total_issues}");
  }

  /**
   * Runs audit using Batch API (safe for large sites).
   */
  protected function runBatch(array $media_ids): void {
    $batch = [
      'title' => dt('Media Accessibility Audit'),
      'operations' => [],
      'finished' => [self::class, 'batchFinished'],
    ];

    foreach (array_chunk($media_ids, 20) as $chunk) {
      $batch['operations'][] = [
        [self::class, 'batchOperation'],
        [$chunk],
      ];
    }

    batch_set($batch);

    // Drush-safe batch processing.
    drush_backend_batch_process();
  }

  /**
   * Batch operation callback.
   */
  public static function batchOperation(array $mids, array &$context): void {
    /** @var \Drupal\media_accessibility_audit\Service\MediaAccessibilityAuditor $auditor */
    $auditor = \Drupal::service('media_accessibility_audit.auditor');

    $storage = \Drupal::entityTypeManager()->getStorage('media');

    foreach ($storage->loadMultiple($mids) as $media) {
      if (!$media instanceof MediaInterface) {
        continue;
      }

      $issues = $auditor->audit($media);
      if ($issues) {
        $context['results'][] = [
          'id' => $media->id(),
          'issues' => $issues,
        ];
      }
    }

    $context['message'] = dt('Processed @count media items.', [
      '@count' => count($mids),
    ]);
  }

  /**
   * Batch finished callback.
   */
  public static function batchFinished(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();

    if ($success) {
      $messenger->addStatus(
        dt('Media audit completed. Issues found: @count', [
          '@count' => count($results),
        ])
      );
    }
    else {
      $messenger->addError(dt('Media audit did not complete successfully.'));
    }
  }

}
