<?php

namespace Drupal\media_accessibility_audit\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\media\Entity\Media;
use Drupal\Component\Utility\Unicode;
use Drupal\media_accessibility_audit\Service\MediaAccessibilityAuditor;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;

/**
 * Controller for Media Accessibility Audit dashboard and batch.
 */
class MediaAccessibilityAuditController extends ControllerBase {

  /**
   * The auditor service.
   *
   * @var \Drupal\media_accessibility_audit\Service\MediaAccessibilityAuditor
   */
  protected MediaAccessibilityAuditor $auditor;

  /**
   * The file url generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  public function __construct(
    MediaAccessibilityAuditor $auditor,
    FileUrlGeneratorInterface $fileUrlGenerator,
    RequestStack $requestStack,
    RendererInterface $renderer,
  ) {
    $this->auditor = $auditor;
    $this->fileUrlGenerator = $fileUrlGenerator;
    $this->requestStack = $requestStack;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
          $container->get('media_accessibility_audit.auditor'),
          $container->get('file_url_generator'),
          $container->get('request_stack'),
          $container->get('renderer')
      );
  }

  /**
   * Media audit dashboard.
   */
  public function dashboard(): array {
    $header = [
      'select' => [
        'data' => [
          '#type' => 'html_tag',
          '#tag' => 'input',
          '#attributes' => [
            'type' => 'checkbox',
            'id' => 'select-all-media',
          ],
        ],
      ],
      'id' => $this->t('MID'),
      'thumbnail' => $this->t('Preview'),
      'status' => $this->t('Status'),
      'score' => $this->t('Score'),
      'issues' => $this->t('Issues'),
      'suggestion' => $this->t('Suggested Alt Text'),
      'apply' => $this->t('Action'),
    ];

    $rows = [];

    // Run batch audit button.
    $batch_url = Url::fromRoute('media_accessibility_audit.batch_audit');
    $batch_link = Link::fromTextAndUrl($this->t('Run Batch Audit'), $batch_url)->toRenderable();
    $batch_link['#attributes'] = ['class' => ['button', 'button--primary']];

    // Load all media entities.
    $limit = 15;
    $media_ids = $this->entityTypeManager()
      ->getStorage('media')
      ->getQuery()
      ->accessCheck(FALSE)
      ->sort('changed', 'DESC')
      ->pager($limit)
      ->execute();

    foreach ($media_ids as $mid) {
      /** @var \Drupal\media\MediaInterface|null $media */
      $media = $this->entityTypeManager
        ->getStorage('media')
        ->load($mid);
      if (!$media) {
        continue;
      }

      // 1. Always read STORED alt text (never runtime-mutated values).
      $stored_alt = $this->auditor->getStoredAlt($media);

      // 2. Determine quality based only on stored value.
      $quality = $this->auditor->getAltQuality($stored_alt);

      // 3. Build issues list from quality (NOT from AI suggestion).
      $issues = [];
      // Add the original Alt Tag info at the top.
      // If the alt is empty, provide a fallback string like 'None'.
      $display_alt = !empty($stored_alt) ? $stored_alt : $this->t('missing');
      $issues[] = '<span class="issue-alt">' . $this->t('Alt tag:') . $display_alt . '</span>';

      if ($quality === 'missing') {
        $issues[] = 'Alt text is missing.';
      }
      elseif ($quality === 'junk') {
        $issues[] = 'Alt text is low quality.';
      }
      elseif ($quality === 'generic') {
        $issues[] = 'Alt text is generic.';
      }
      else {
        $issues[] = 'No issues found.';
      }

      // 4. Status + score derived ONLY from stored alt.
      $status_class = match ($quality) {
        // Red.
        'missing', 'junk' => 'badge-fail',
        // Yellow/Orange.
        'generic' => 'badge-warning',
        // Green.
        'good', 'pass' => 'badge-success',
        // Grey.
        default => 'badge-secondary',
      };
      $status_label = match ($quality) {
        'missing', 'junk' => $this->t('Fail'),
        'generic' => $this->t('Warning'),
        default => $this->t('Pass'),
      };
      $status_badge = [
        '#type' => 'html_tag',
        '#tag' => 'span',
        '#value' => $status_label,
        '#attributes' => [
          'class' => ['badge', $status_class],
          'style' => 'min-width: 60px; text-align: center; display: inline-block;',
        ],
      ];
      $score = match ($quality) {
        'missing' => 10,
        'junk' => 30,
        'generic' => 60,
        'good' => 95,
        default => 50,
      };

      $suggestion_data = $this->auditor->suggestAltText($media);

      $alt_text = $suggestion_data['alt_text'] ?? '';
      $vision_used = $suggestion_data['vision_used'] ?? FALSE;
      $confidence = $suggestion_data['confidence'] ?? 'low';

      // Apply link (only if alt text exists)
      $apply_link = '';
      if (!empty($alt_text)) {
        $apply_link = Link::fromTextAndUrl(
              $this->t('Apply'),
              Url::fromRoute('media_accessibility_audit.apply_suggestion', ['media' => $mid])
          )->toRenderable();
        $apply_link['#attributes'] = ['class' => ['button', 'button--small']];
      }

      // Define colors for confidence levels.
      $confidence_class = match ($confidence) {
        'high' => 'badge-success',
        'medium' => 'badge-warning',
        default => 'badge-secondary',
      };

      // Suggestion render array with badges.
      $suggestion_render = [
        'data' => [
          '#type' => 'container',
          '#attributes' => [
            'id' => 'suggestion-media-' . $mid,
            'class' => ['suggestion-wrapper'],
          ],
          'text' => [
            '#markup' => $alt_text . ' ',
          ],
          'refresh' => [
            '#type' => 'link',
            '#title' => [
              '#markup' => '<span class="maa-refresh-icon" aria-hidden="true">↻</span>',
            ],
            '#url' => Url::fromRoute(
            'media_accessibility_audit.refresh_suggestion_ajax',
            ['media' => $mid]
            ),
            '#attributes' => [
              'class' => ['refresh-suggestion-ajax'],
              'title' => $this->t('Refresh suggestion'),
              'aria-label' => $this->t('Refresh suggestion'),
              'style' => 'margin-left:8px; cursor:pointer;',
            ],
          ],
          'loader' => [
            '#type' => 'html_tag',
            '#tag' => 'span',
            '#attributes' => [
              'class' => ['suggestion-loader'],
              'aria-hidden' => 'true',
            ],
          ],
          'thinking' => [
            '#type' => 'html_tag',
            '#tag' => 'span',
            '#value' => $this->t('AI is thinking…'),
            '#attributes' => [
              'class' => ['suggestion-thinking'],
              'aria-live' => 'polite',
            ],
          ],
          'badges_wrapper' => [
            '#type' => 'container',
            '#attributes' => [
              'class' => [
                'suggestion-badges', 'clearfix',
              ],
            ],
            'vision' => $vision_used ? [
              '#type' => 'html_tag',
              '#tag' => 'span',
              '#value' => '🖼️ ' . $this->t('Vision'),
              '#attributes' => [
                'class' => [
                  'badge', 'badge-success', 'badge-vision',
                ],
              ],
            ] : [],
            'confidence' => [
              '#type' => 'html_tag',
              '#tag' => 'span',
              '#value' => ucfirst($confidence),
              '#attributes' => [
                'class' => ['badge', $confidence_class],
              ],
            ],
          ],
        ],
      ];

      // Get first image field dynamically.
      $thumbnail_column = '';
      $image_fields = array_filter(
            $media->getFieldDefinitions(),
            fn($definition) => $definition->getType() === 'image'
        );

      foreach ($image_fields as $field_name => $definition) {
        $items = $media->get($field_name);
        if (!$items->isEmpty()) {
          $file = $items->first()->entity;
          if ($file && $file->getFileUri()) {

            // Get the full filename.
            $full_filename = $file->getFilename();
            $public_url_string = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
            $url = Url::fromUri($public_url_string);

            // Truncate the filename for display purposes only.
            $display_filename = Unicode::truncate(
              $full_filename,
              20,
              TRUE,
              '...'
              );

            $image_render = [
              '#theme' => 'image_style',
              '#style_name' => 'thumbnail',
              '#uri' => $file->getFileUri(),
              '#alt' => $media->label(),
              '#prefix' => '<div class="media-auditor-thumbnail">',
              '#suffix' => '</div>',
            ];

            $media_link = Link::fromTextAndUrl($display_filename, $url)->toRenderable();
            $media_link['#attributes'] = [
              'style' => 'display:block; font-weight:bold; margin-top:5px;',
              'title' => $full_filename,
            ];

            $edit_link = $media->toLink($this->t('Edit Media'), 'edit-form', [
              'attributes' => [
                'class' => [
                  'button', 'button--extrasmall',
                ],
                'style' => 'font-size:11px; margin-top:5px;',
              ],
            ])->toRenderable();

            $thumbnail_column = [
              'data' => [
                'image' => $image_render,
                'filename' => $media_link,
                'edit_action' => $edit_link,
              ],
            ];
            break;
          }
        }
      }

      $rows[] = [
        'select' => [
          'data' => [
            '#type' => 'html_tag',
            '#tag' => 'input',
            '#attributes' => [
              'type' => 'checkbox',
              'class' => ['media-select'],
              'data-media-id' => (string) $mid,
            ],
          ],
        ],
        'id' => $mid,
        'thumbnail' => $thumbnail_column,
        'status' => ['data' => $status_badge],
        'score' => $score,
        'issues' => [
          'data' => [
            '#markup' => implode('<br>', $issues),
          ],
        ],
        'suggestion' => $suggestion_render,
        'apply' => ['data' => $apply_link],
      ];
    }

    $bulk_apply_button = [
      '#type' => 'html_tag',
      '#tag' => 'button',
      '#value' => $this->t('Apply selected suggestions'),
      '#attributes' => [
        'id' => 'bulk-apply-btn',
        'class' => ['button', 'button--primary'],
        'disabled' => 'disabled',
        'style' => 'margin-left:10px;',
      ],
    ];

    return [
      '#type' => 'container',
      '#attached' => [
        'library' => [
          'media_accessibility_audit/dashboard',
        ],
      ],
      'actions' => [
        '#type' => 'container',
        'batch' => $batch_link,
        'bulk_apply' => $bulk_apply_button,
      ],
      'table' => [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No media items found.'),
        '#attributes' => ['class' => ['media-accessibility-table']],
      ],
      'pager' => [
        '#type' => 'pager',
      ],
    ];
  }

  /**
   * Run batch audit from browser.
   */
  public function runBatchAudit() {
    $media_ids = $this->entityTypeManager()
      ->getStorage('media')
      ->getQuery()
      ->accessCheck(FALSE)
      ->execute();

    if (empty($media_ids)) {
      $this->messenger()->addWarning($this->t('No media items found.'));
      return $this->redirect('media_accessibility_audit.dashboard');
    }

    $batch = [
      'title' => $this->t('Media Accessibility Audit'),
      'operations' => [],
      'finished' => [self::class, 'batchFinished'],
      'init_message' => $this->t('Starting media accessibility audit…'),
      'progress_message' => $this->t('Processed @current of @total media items.'),
      'error_message' => $this->t('An error occurred during the audit.'),
    ];

    foreach (array_chunk($media_ids, 20) as $chunk) {
      $batch['operations'][] = [
            [self::class, 'batchOperation'],
            [$chunk],
      ];
    }

    // phpcs:ignore Drupal.Semantics.LackingDependencyInjection
    batch_set($batch);

    // phpcs:ignore Drupal.Semantics.LackingDependencyInjection
    return batch_process('/admin/content/media-accessibility-audit');
  }

  /**
   * Batch operation callback (static for Batch API).
   */
  public static function batchOperation(array $mids, array &$context): void {
    // phpcs:ignore Drupal.Semantics.LackingDependencyInjection
    $auditor = \Drupal::service('media_accessibility_audit.auditor');

    $context['results']['processed'] ??= 0;
    $context['results']['issues'] ??= [];

    foreach ($mids as $mid) {
      $context['results']['processed']++;

      // phpcs:ignore Drupal.Semantics.LackingDependencyInjection
      $media = \Drupal::entityTypeManager()
        ->getStorage('media')
        ->load($mid);

      if (!$media) {
        continue;
      }

      $issues = $auditor->audit($media);

      if (!empty($issues)) {
        $context['results']['issues'][] = [
          'id' => $mid,
          'issues' => $issues,
        ];
      }
    }

    // phpcs:ignore Drupal.Semantics.LackingDependencyInjection
    $context['message'] = t('Processed @count media items.', ['@count' => count($mids)]);
  }

  /**
   * Batch finished callback (static).
   */
  public static function batchFinished($success, $results, $operations): void {
    $messenger = \Drupal::messenger();

    if (!$success) {
      // phpcs:ignore Drupal.Semantics.LackingDependencyInjection
      $messenger->addError(t('Media audit did not complete successfully.'));
      return;
    }

    $processed = $results['processed'] ?? 0;
    $issues = $results['issues'] ?? [];

    if (empty($issues)) {
      // phpcs:ignore Drupal.Semantics.LackingDependencyInjection
      $messenger->addStatus(
      t('Audit completed. @count media items checked. No accessibility issues found.', [
        '@count' => $processed,
      ])
      );
      return;
    }

    // phpcs:ignore Drupal.Semantics.LackingDependencyInjection
    $messenger->addWarning(
    t(
      'Audit completed. @issues issues found across @media media items.',
      [
        '@issues' => count($issues),
        '@media' => $processed,
      ]
    )
    );
  }

  /**
   * Apply Suggestion from AI for media items.
   */
  public function applySuggestion(Media $media) {
    $suggestion_data = $this->auditor->suggestAltText($media);
    $suggestion_text = is_array($suggestion_data) ? ($suggestion_data['alt_text'] ?? '') : $suggestion_data;

    if (empty($suggestion_text)) {
      $this->messenger()->addWarning($this->t('No suggestion available.'));
      return $this->redirect('media_accessibility_audit.dashboard');
    }

    // Standard media image field is usually 'field_media_image'.
    if ($media->hasField('field_media_image')) {
      $image_field = $media->get('field_media_image');

      // Update the actual source field using an array
      // to ensure all properties are saved.
      $media->set('field_media_image', [
        'target_id' => $image_field->target_id,
        'alt' => (string) $suggestion_text,
        'title' => $image_field->title,
      ]);

      // IMPORTANT: Manually update thumbnail metadata to prevent UI desync.
      $media->set('thumbnail', [
        'target_id' => $media->get('thumbnail')->target_id,
        'alt' => (string) $suggestion_text,
      ]);

      // Save the media entity.
      $media->save();

      $this->messenger()->addStatus($this->t('Successfully updated alt text.'));
    }

    return $this->redirect('media_accessibility_audit.dashboard');
  }

  /**
   * Refresh AI suggestion for a media item.
   *
   * This does NOT save anything.
   * It simply forces a new suggestion generation.
   */
  public function refreshSuggestion(Media $media) {
    // Force re-generation (no cache yet).
    $this->auditor->suggestAltText($media, ['force_refresh' => TRUE]);

    $this->messenger()->addStatus(
    $this->t('Suggestion refreshed.')
    );

    return $this->redirect('media_accessibility_audit.dashboard');
  }

  /**
   * Bulk apply AI-generated alt text to selected media.
   */
  public function bulkApply() {
    $request = $this->requestStack->getCurrentRequest();
    $ids_param = $request->query->get('ids');
    if (empty($ids_param)) {
      $this->messenger()->addWarning($this->t('No media selected.'));
      return $this->redirect('media_accessibility_audit.dashboard');
    }

    $ids = array_filter(array_map('intval', explode(',', $ids_param)));
    $applied = 0;

    foreach ($ids as $mid) {
      /** @var \Drupal\media\MediaInterface|null $media */
      $media = $this->entityTypeManager()
        ->getStorage('media')
        ->load($mid);

      if (!$media) {
        continue;
      }

      $suggestion_data = $this->auditor->suggestAltText($media);
      $alt_text = $suggestion_data['alt_text'] ?? '';

      if (empty($alt_text)) {
        continue;
      }

      if ($media->hasField('field_media_image')) {
        $image_field = $media->get('field_media_image');

        $media->set('field_media_image', [
          'target_id' => $image_field->target_id,
          'alt' => $alt_text,
          'title' => $image_field->title,
        ]);

        $media->set('thumbnail', [
          'target_id' => $media->get('thumbnail')->target_id,
          'alt' => $alt_text,
        ]);

        $media->save();
        $applied++;
      }
    }

    $this->messenger()->addStatus(
    $this->t('Applied alt text to @count media items.', ['@count' => $applied])
    );

    return $this->redirect('media_accessibility_audit.dashboard');
  }

  /**
   * Refresh suggestion method.
   */
  public function refreshSuggestionAjax(Media $media): JsonResponse {
    $suggestion_data = $this->auditor->suggestAltText($media, ['force_refresh' => TRUE]);

    $alt_text = $suggestion_data['alt_text'] ?? '';
    $vision_used = $suggestion_data['vision_used'] ?? FALSE;
    $confidence = $suggestion_data['confidence'] ?? 'low';

    $confidence_class = match ($confidence) {
      'high' => 'badge-success',
      'medium' => 'badge-warning',
      default => 'badge-secondary',
    };

    $render = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['suggestion-wrapper'],
        'id' => 'suggestion-media-' . $media->id(),
      ],
      'text' => [
        '#markup' => $alt_text . ' ',
      ],
      'refresh' => [
        '#type' => 'link',
        '#title' => [
          '#markup' => '<span class="maa-refresh-icon" aria-hidden="true">↻</span>',
        ],
        '#url' => Url::fromRoute(
            'media_accessibility_audit.refresh_suggestion_ajax',
            ['media' => $media->id()]
        ),
        '#attributes' => [
          'class' => ['refresh-suggestion-ajax'],
          'title' => $this->t('Refresh suggestion'),
          'aria-label' => $this->t('Refresh suggestion'),
          'style' => 'margin-left:8px; cursor:pointer;',
        ],
      ],
      'thinking' => [
        '#type' => 'html_tag',
        '#tag' => 'span',
        '#value' => $this->t('AI is thinking…'),
        '#attributes' => [
          'class' => ['suggestion-thinking'],
          'aria-live' => 'polite',
        ],
      ],
      'badges_wrapper' => [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'suggestion-badges', 'clearfix',
          ],
        ],
        'vision' => $vision_used ? [
          '#type' => 'html_tag',
          '#tag' => 'span',
          '#value' => '🖼️ ' . $this->t('Vision'),
          '#attributes' => [
            'class' => [
              'badge', 'badge-success', 'badge-vision',
            ],
          ],
        ] : [],
        'confidence' => [
          '#type' => 'html_tag',
          '#tag' => 'span',
          '#value' => ucfirst($confidence),
          '#attributes' => [
            'class' => ['badge', $confidence_class],
          ],
        ],
      ],
    ];

    $html = $this->renderer->renderRoot($render);

    return new JsonResponse([
      'html' => $html,
      'target' => 'suggestion-media-' . $media->id(),
    ]);
  }

}
