<?php

declare(strict_types=1);

namespace Drupal\media_accessibility_audit\Service;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\media\MediaInterface;
use Drupal\node\NodeInterface;
use Psr\Log\LoggerInterface;

/**
 * Audits media entities and generates accessibility recommendations.
 *
 * This service:
 * - Evaluates stored alt text quality
 * - Suggests alt text via AI (optionally vision-enabled)
 * - Scores accessibility confidence.
 *
 * No entity mutation happens here.
 * This service is READ-ONLY by design.
 */
final class MediaAccessibilityAuditor {

  public function __construct(
    protected AiProviderPluginManager $aiProviderManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected VisionProviderService $visionProvider,
    protected MessengerInterface $messenger,
    protected LoggerInterface $logger,
  ) {}

  /**
   * Audits a media entity for accessibility issues.
   *
   * @return string[]
   *   Human-readable issue descriptions.
   */
  public function audit(MediaInterface $media): array {
    $issues = [];

    if ($media->bundle() !== 'image') {
      return $issues;
    }

    if (!$this->getFirstImage($media)) {
      return ['Missing image file.'];
    }

    $alt = $this->getStoredAlt($media);
    $quality = $this->getAltQuality($alt);

    match ($quality) {
      'missing' => $issues[] = 'Alt text is missing.',
      'junk' => $issues[] = 'Alt text is present but meaningless.',
      'generic' => $issues[] = 'Alt text is too generic.',
      default => NULL,
    };

    return $issues;
  }

  /**
   * Generates an AI-based alt text suggestion.
   *
   * This method NEVER saves data.
   * Consumers must explicitly apply the suggestion.
   */
  public function suggestAltText(MediaInterface $media, array $context = []): array {
    $parent_node = $this->findParentNode($media);
    $parent_title = $parent_node?->label();

    $subject_hint = $parent_title
      ? sprintf('The image is used in an article titled "%s".', $parent_title)
      : 'The image is used in a general web context.';

    $vision_available = $this->visionProvider->providerSupportsVision();

    $prompt = sprintf(
      'You are an accessibility expert. Write a concise, WCAG-compliant alt text (max 80 characters) for the attached image. %s',
      $subject_hint
    );

    $ai_text = '';

    try {
      $defaults = $this->aiProviderManager->getDefaultProviderForOperationType('chat');
      $provider = $this->aiProviderManager->createInstance($defaults['provider_id']);

      $images = [];
      if ($vision_available && $media->hasField('field_media_image')) {
        $file = $media->get('field_media_image')->entity;
        if ($file) {
          $image = new ImageFile();
          $image->setFileFromFile($file);
          $images[] = $image;
        }
      }

      $input = new ChatInput([
        new ChatMessage('user', $prompt, $images),
      ]);

      $response = $provider->chat($input, $defaults['model_id']);
      $ai_text = trim($response->getNormalized()->getText());

      if ($this->looksLikeRefusal($ai_text)) {
        $ai_text = '';
      }

    }
    catch (\Throwable $e) {
      if (str_contains($e->getMessage(), 'quota')) {
        $this->messenger->addWarning('AI quota exceeded. Please try again shortly.');
      }
      $this->logger->error('Alt suggestion failed: @message', ['@message' => $e->getMessage()]);
    }

    $confidence = 'low';

    if ($ai_text === '') {
      $ai_text = $parent_title
        ? sprintf('Illustrative image supporting %s', $parent_title)
        : 'Illustrative image related to the content';
    }
    else {
      $confidence = $vision_available ? 'high' : 'medium';
    }

    return [
      'alt_text' => $ai_text,
      'vision_used' => $vision_available,
      'confidence' => $confidence,
    ];
  }

  /**
   * Returns stored alt text from the media image field.
   */
  public function getStoredAlt(MediaInterface $media): string {
    return $media->hasField('field_media_image') && !$media->get('field_media_image')->isEmpty()
      ? (string) ($media->get('field_media_image')->first()->alt ?? '')
      : '';
  }

  /**
   * Classifies alt text quality.
   */
  public function getAltQuality(string $alt): string {
    $alt = trim(mb_strtolower($alt));

    if ($alt === '') {
      return 'missing';
    }

    if (mb_strlen($alt) < 8 || in_array($alt, ['img', 'image', 'photo', 'test', 'n/a'], TRUE)) {
      return 'junk';
    }

    if (str_contains($alt, 'illustrative image')) {
      return 'generic';
    }

    return 'good';
  }

  /**
   * Returns first populated image field item.
   */
  public function getFirstImage(MediaInterface $media): ?FieldItemInterface {
    foreach ($media->getFieldDefinitions() as $field_name => $definition) {
      if ($definition->getType() === 'image' && !$media->get($field_name)->isEmpty()) {
        return $media->get($field_name)->first();
      }
    }
    return NULL;
  }

  /**
   * Finds the first node referencing this media entity.
   */
  private function findParentNode(MediaInterface $media): ?NodeInterface {
    $storage = $this->entityTypeManager->getStorage('node');
    $field_map = $this->entityFieldManager->getFieldMapByFieldType('entity_reference');

    foreach ($field_map['node'] ?? [] as $field_name => $info) {
      if (!in_array('media', $info['target_types'] ?? [], TRUE)) {
        continue;
      }

      $ids = $storage->getQuery()
        ->accessCheck(FALSE)
        ->condition($field_name, $media->id())
        ->range(0, 1)
        ->execute();

      if ($ids) {
        return $storage->load(reset($ids));
      }
    }

    return NULL;
  }

  /**
   * Detects common AI refusal / apology responses.
   */
  private function looksLikeRefusal(string $text): bool {
    return stripos($text, 'cannot') !== FALSE || stripos($text, 'sorry') !== FALSE;
  }

}
