<?php

declare(strict_types=1);

namespace Drupal\media_accessibility_audit\Service;

use Drupal\ai\AiProviderPluginManager;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\media\MediaInterface;
use Psr\Log\LoggerInterface;

/**
 * Provides vision-capability checks for the active AI provider.
 *
 * This service is intentionally lightweight:
 * - No AI execution
 * - No entity loading
 * - Pure capability & helper logic.
 */
final class VisionProviderService {

  /**
   * Logger channel for media accessibility audit.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs the VisionProviderService.
   *
   * @param \Drupal\ai\AiProviderPluginManager $aiProviderManager
   *   The AI provider plugin manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger channel factory.
   */
  public function __construct(
    protected AiProviderPluginManager $aiProviderManager,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->logger = $loggerFactory->get('media_accessibility_audit');
  }

  /**
   * Determines whether the active AI model supports vision input.
   *
   * @return bool
   *   TRUE if vision/image input is supported.
   */
  public function providerSupportsVision(): bool {
    try {
      $defaults = $this->aiProviderManager
        ->getDefaultProviderForOperationType('chat');

      $model_id = $defaults['model_id'] ?? '';

      // Known vision-capable models.
      $vision_models = [
        'gpt-4o',
        'claude-3-5-sonnet',
        'claude-3-7-sonnet',
        'gemini-1.5-pro',
        'chat_with_image_vision',
      ];

      return in_array($model_id, $vision_models, TRUE);
    }
    catch (\Throwable $e) {
      $this->logger->error(
        'Vision capability check failed: @message',
        ['@message' => $e->getMessage()]
      );
      return FALSE;
    }
  }

  /**
   * Returns the file URI of the first image attached to a media entity.
   *
   * @param \Drupal\media\MediaInterface $media
   *   Media entity.
   *
   * @return string|null
   *   File URI if found, NULL otherwise.
   */
  public function getFirstImageUri(MediaInterface $media): ?string {
    $image_item = $this->getFirstImageItem($media);
    $file = $image_item?->entity;

    return $file?->getFileUri();
  }

  /**
   * Returns the first populated image field item on the media entity.
   *
   * @param \Drupal\media\MediaInterface $media
   *   Media entity.
   *
   * @return \Drupal\Core\Field\FieldItemInterface|null
   *   Image field item or NULL.
   */
  protected function getFirstImageItem(MediaInterface $media) {
    foreach ($media->getFieldDefinitions() as $field_name => $definition) {
      if (
        $definition->getType() === 'image'
        && !$media->get($field_name)->isEmpty()
      ) {
        return $media->get($field_name)->first();
      }
    }
    return NULL;
  }

}
