<?php

namespace Drupal\media_accessibility_enhancer\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'video_transcription' field type.
 *
 * @FieldType(
 *   id = "video_transcription",
 *   label = @Translation("Video Transcription"),
 *   description = @Translation("Store video transcription with timestamp support"),
 *   default_widget = "video_transcription_widget",
 *   default_formatter = "video_transcription_formatter"
 * )
 */
class VideoTranscriptionItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties['transcription'] = DataDefinition::create('string')
      ->setLabel(t('Transcription text'))
      ->setRequired(FALSE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'transcription' => [
          'type' => 'text',
          'size' => 'big',
          'not null' => FALSE,
        ],
      ],
      'indexes' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $value = $this->get('transcription')->getValue();
    return $value === NULL || $value === '';
  }

  /**
   * Get word count of transcription.
   *
   * @return int
   *   Word count.
   */
  public function getWordCount() {
    $text = $this->get('transcription')->getValue();
    if (empty($text)) {
      return 0;
    }
    return str_word_count(strip_tags($text));
  }

  /**
   * Get character count of transcription.
   *
   * @return int
   *   Character count.
   */
  public function getCharCount() {
    $text = $this->get('transcription')->getValue();
    if (empty($text)) {
      return 0;
    }
    return mb_strlen(strip_tags($text));
  }

}
