<?php

namespace Drupal\media_accessibility_enhancer\Plugin\Field\FieldWidget;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\media\OEmbed\ProviderException;
use Drupal\media\OEmbed\ResourceException;
use Drupal\media\OEmbed\UrlResolverInterface;
use Drupal\media_accessibility_enhancer\Plugin\MediaDistantProviderPluginManager;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'video_duration_widget' widget.
 *
 * @FieldWidget(
 *   id = "video_duration_widget",
 *   label = @Translation("Video Duration"),
 *   field_types = {
 *     "video_duration"
 *   }
 * )
 */
class VideoDurationWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    private UrlResolverInterface $urlResolver,
    private LoggerChannelFactoryInterface $logger,
    private MediaDistantProviderPluginManager $mediaDistantProviderPluginManager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('media.oembed.url_resolver'),
      $container->get('logger.factory'),
      $container->get('plugin.manager.media_distant_provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $wrapper = 'ajax-wrapper-duration';

    $total_seconds = isset($items[$delta]->seconds) ? $items[$delta]->seconds : 0;
    $hours = floor($total_seconds / 3600);
    $minutes = floor(($total_seconds % 3600) / 60);
    $seconds = $total_seconds % 60;

    $element['duration'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form--inline']],
    ];

    $element['duration']['hours'] = [
      '#type' => 'number',
      '#title' => $this->t('Hours'),
      '#default_value' => $hours,
      '#min' => 0,
      '#step' => 1,
      '#prefix' => '<div id="' . $wrapper . '">',
    ];

    $element['duration']['minutes'] = [
      '#type' => 'number',
      '#title' => $this->t('Minutes'),
      '#default_value' => $minutes,
      '#min' => 0,
      '#max' => 59,
      '#step' => 1,
    ];

    $element['duration']['seconds'] = [
      '#type' => 'number',
      '#title' => $this->t('Seconds'),
      '#default_value' => $seconds,
      '#min' => 0,
      '#max' => 59,
      '#step' => 1,
      '#suffix' => '</div>',
    ];

    $media = $items->getEntity();
    if ($media && $media->hasField('field_media_oembed_video') && !empty($media->get('field_media_oembed_video')->value)) {
      $url = $media->get('field_media_oembed_video')->value;

      try {
        $urlResolverResult = $this->urlResolver->getProviderByUrl($url);
      } catch (ProviderException|ResourceException $exception) {
        $this->logger->get('media_accessibility_enhancer')->alert($exception->getMessage());
      }

      if (isset($urlResolverResult)) {
        foreach ($this->mediaDistantProviderPluginManager->getDefinitions() as $key => $definition) {
          if (isset($definition['oembed_provider']) && $definition['oembed_provider'] === $urlResolverResult->getName()) {
            $element['provide_' . $key . '_duration'] = [
              '#type' => 'button',
              '#value' => $this->t('Call @plugin for populated field', ['@plugin' => $definition['label']]),
              '#name' => 'provide_duration',
              '#ajax' => [
                'callback' => [$this, 'ajaxProviderCallbackDuration'],
                'event' => 'click',
                'wrapper' => $wrapper,
              ],
              '#plugin_key' => $key,
              '#url' => $url,
              '#action' => 'duration'
            ];
          }
        }
      }
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    foreach ($values as &$value) {
      if (isset($value['duration'])) {
        $hours = (int) ($value['duration']['hours'] ?? 0);
        $minutes = (int) ($value['duration']['minutes'] ?? 0);
        $seconds = (int) ($value['duration']['seconds'] ?? 0);

        $value['seconds'] = ($hours * 3600) + ($minutes * 60) + $seconds;

        unset($value['duration']);
      }
    }
    return $values;
  }

  /**
   * Ajax callback for set value.
   *
   * @param array $form
   *   This is current form.
   * @param FormStateInterface $form_state
   *   This is form state.
   *
   * @return AjaxResponse
   *   Return ajax response.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function ajaxProviderCallbackDuration(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $plugin_key = $triggering_element['#plugin_key'];
    $url = $triggering_element['#url'];
    $action = $triggering_element['#action'];

    if ($action === 'duration') {
      // Access the plugin instance.
      $plugin_instance = $this->mediaDistantProviderPluginManager->createInstance($plugin_key);

      // Fetch duration (make sure your plugin has this method!).
      if (method_exists($plugin_instance, 'getDuration') && isset($url)) {
        $resource = $plugin_instance->getResourceInformation($url);
        $duration = $plugin_instance->getDuration($resource, $url);
      } else {
        $duration = 0;
      }

      $hours = floor($duration / 3600);
      $minutes = floor(($duration % 3600) / 60);
      $seconds = $duration % 60;

      // Update the field value.
      $response = new AjaxResponse();
      $response->addCommand(new InvokeCommand('#edit-field-video-duration-0-duration-hours', 'val', [$hours]));
      $response->addCommand(new InvokeCommand('#edit-field-video-duration-0-duration-minutes', 'val', [$minutes]));
      $response->addCommand(new InvokeCommand('#edit-field-video-duration-0-duration-seconds', 'val', [$seconds]));

      return $response;
    }
  }

}
