<?php

namespace Drupal\media_accessibility_enhancer\Plugin\Field\FieldWidget;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\media\OEmbed\ProviderException;
use Drupal\media\OEmbed\ResourceException;
use Drupal\media\OEmbed\UrlResolverInterface;
use Drupal\media_accessibility_enhancer\Plugin\MediaDistantProviderPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'video_transcription_widget' widget.
 *
 * @FieldWidget(
 *   id = "video_transcription_widget",
 *   label = @Translation("Video Transcription"),
 *   field_types = {
 *     "video_transcription"
 *   }
 * )
 */
class VideoTranscriptionWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    private UrlResolverInterface $urlResolver,
    private LoggerChannelFactoryInterface $logger,
    private MediaDistantProviderPluginManager $mediaDistantProviderPluginManager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('media.oembed.url_resolver'),
      $container->get('logger.factory'),
      $container->get('plugin.manager.media_distant_provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $wrapper = 'ajax-wrapper-transcription';
    $element['transcription'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Transcription'),
      '#default_value' => isset($items[$delta]->transcription) ? $items[$delta]->transcription : NULL,
      '#rows' => 10,
      '#description' => $this->t('Enter the video transcription text.'),
      '#prefix' => '<div id="' . $wrapper . '">',
      '#suffix' => '</div>',
    ];

    $media = $items->getEntity();
    if ($media && $media->hasField('field_media_oembed_video') && !empty($media->get('field_media_oembed_video')->value)) {
      $url = $media->get('field_media_oembed_video')->value;

      try {
        $urlResolverResult = $this->urlResolver->getProviderByUrl($url);
      } catch (ProviderException|ResourceException $exception) {
        $this->logger->get('media_accessibility_enhancer')->alert($exception->getMessage());
      }

      if (isset($urlResolverResult)) {
        foreach ($this->mediaDistantProviderPluginManager->getDefinitions() as $key => $definition) {
          if (isset($definition['oembed_provider']) && $definition['oembed_provider'] === $urlResolverResult->getName()) {
            $element['provide_' . $key. '_transcription'] = [
              '#type' => 'button',
              '#value' => $this->t('Call @plugin for populated field', ['@plugin' => $definition['label']]),
              '#name' => 'provide_transcription',
              '#ajax' => [
                'callback' => [$this, 'ajaxProviderCallbackTranscription'],
                'event' => 'click',
                'wrapper' => $wrapper,
              ],
              '#plugin_key' => $key,
              '#url' => $url,
              '#action' => 'transcription',
              '#language' => $media->language()->getId(),
            ];
          }
        }
      }
    }

    return $element;
  }

  /**
   * Ajax callback for set value.
   *
   * @param array $form
   *   This is current form.
   * @param FormStateInterface $form_state
   *   This is form state.
   *
   * @return AjaxResponse
   *   Return ajax response.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function ajaxProviderCallbackTranscription(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $plugin_key = $triggering_element['#plugin_key'];
    $url = $triggering_element['#url'];
    $action = $triggering_element['#action'];
    $language = $triggering_element['#language'];

    if ($action === 'transcription') {
      // Access the plugin instance.
      $plugin_instance = $this->mediaDistantProviderPluginManager->createInstance($plugin_key);

      // Fetch duration (make sure your plugin has this method!).
      if (method_exists($plugin_instance, 'getTranscription') && isset($url)) {
        $resource = $plugin_instance->getResourceInformation($url);
        $transcription = $plugin_instance->getTranscription($resource, $url, $language);
      } else {
        $transcription = 0;
      }

      // Update the seconds field value.
      $response = new AjaxResponse();
      $selector = '#edit-field-video-transcription-0-transcription';
      $response->addCommand(new InvokeCommand($selector, 'val', [$transcription]));

      return $response;
    }
  }

}
