<?php

namespace Drupal\media_accessibility_enhancer\Plugin\MediaDistantProvider;

use Drupal\media\OEmbed\Resource;
use Drupal\media_accessibility_enhancer\Plugin\MediaDistantProviderBase;

/**
 * @MediaDistantProvider(
 *   id = "vimeo_media_distant_provider",
 *   label = @Translation("Vimeo Media Distant Provider"),
 *   oembed_provider = "Vimeo"
 * )
 */
class VimeoMediaDistantProvider extends MediaDistantProviderBase {

  /**
   * {@inheritdoc}
   */
  public function getDuration(?Resource $resource, string $url): string {
    $videoId = $this->getVimeoVideoId($url);
    return $this->fetchVimeoDuration($videoId);
  }

  /**
   * {@inheritdoc}
   */
  public function getTranscription(?Resource $resource, string $url, string $language): string {
    $videoId = $this->getVimeoVideoId($url);
    $transcriptionsList = $this->fetchVimeoTranscriptions($videoId);
    $transcriptions = '';

    if (!empty($transcriptionsList)) {
      foreach ($transcriptionsList as $transcription) {
        if ($transcription['lang'] === $language) {
          $transcriptions = $this->vttToText($this->fetchVimeoTranscriptionVtt($transcription['url']));
        }
      }
    } else {
      return $this->t('No transcription found');
    }
    return $transcriptions;
  }

  /**
   * Fetch vimeo duration
   *
   * @param string $videoId
   *   The video id
   * @return mixed|null
   *   Return duration in seconds.
   */
  function fetchVimeoDuration(string $videoId) {
    try {
      $res = $this->httpClient->request('GET', "https://player.vimeo.com/video/$videoId/config");

      $data = json_decode($res->getBody(), TRUE);
      return $data['video']['duration'] ?? null;
    }
    catch (\Throwable $e) {
      $this->logger->get('media_accessibility_enhancer')->error('Vimeo fetch error: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Fetch vimeo transcriptions
   *
   * @param string $videoId
   *   The video id
   * @return mixed|null
   *   Return duration in seconds.
   */
  function fetchVimeoTranscriptions(string $videoId) {
    try {
      $res = $this->httpClient->request('GET', "https://player.vimeo.com/video/$videoId/config");

      $data = json_decode($res->getBody(), TRUE);
      return $data['request']['text_tracks'] ?? null;
    }
    catch (\Throwable $e) {
      $this->logger->get('media_accessibility_enhancer')->error('Vimeo fetch error: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Fetch vimeo transcriptions vtt
   *
   * @param string $url
   *   The url of vtt
   * @return mixed|null
   *   Return duration in seconds.
   */
  function fetchVimeoTranscriptionVtt(string $url) {
    try {
      $res = $this->httpClient->request('GET', "https://player.vimeo.com$url");
      return $res->getBody()->getContents();
    }
    catch (\Throwable $e) {
      $this->logger->get('media_accessibility_enhancer')->error('Vimeo fetch error: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }


  /**
   * Extract a Vimeo video ID from common URL formats.
   *
   * @param string $url
   *   Video URL.
   *
   * @return ?string
   *   Video ID.
   */
  function getVimeoVideoId(string $url): ?string {
    if (preg_match('/vimeo\.com\/(?:video\/)?(\d+)/', $url, $matches)) {
      return $matches[1];
    }

    return null;
  }

  /**
   * Get text from vtt (transcriptions).
   *
   * @param string $vtt
   *   The current vtt file content.
   *
   * @return string
   *   Return text lisible.
   */
  function vttToText(string $vtt): string {
    // Remove WEBVTT header.
    $vtt = preg_replace('/WEBVTT.*\n/', '', $vtt);

    // Remove cue numbers.
    $vtt = preg_replace('/^\d+\s*$/m', '', $vtt);

    // Remove timestamps.
    $vtt = preg_replace('/^\d{2}:\d{2}:\d{2}\.\d{3} --> \d{2}:\d{2}:\d{2}\.\d{3}.*$/m', '', $vtt);

    // Remove HTML tags.
    $vtt = strip_tags($vtt);

    // Remove extra empty lines.
    $vtt = preg_replace("/\n{2,}/", "\n", $vtt);

    // Trim each line and join with space.
    $lines = array_map('trim', explode("\n", $vtt));
    $lines = array_filter($lines);
    return implode(' ', $lines);
  }

}
