<?php

namespace Drupal\media_accessibility_enhancer\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\media_accessibility_enhancer\Plugin\MediaDistantProviderPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for all provider settings.
 */
class MediaDistantProviderSettingsForm extends ConfigFormBase {

  const CONFIG_ID = 'media_accessibility_enhancer.media_distant_provider_settings';

  /**
   * {@inheritdoc}
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    private MediaDistantProviderPluginManager $mediaDistantProviderPluginManager,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('plugin.manager.media_distant_provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      self::CONFIG_ID,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'media_distant_provider_configuration';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(self::CONFIG_ID);

    foreach ($this->mediaDistantProviderPluginManager->getDefinitions() as $key => $definition) {
      /** @var \Drupal\media_accessibility_enhancer\Plugin\MediaDistantProviderBase $instance */
      $instance = $this->mediaDistantProviderPluginManager->createInstance($key);
      $settingsForm = $instance->defineSettings($config->get($key) ?? NULL);
      if (!empty($settingsForm)) {
        $form[$key] = [
          '#type' => 'details',
          '#title' => $instance->getPluginDefinition()['label'],
          '#tree' => TRUE,
          '#open' => TRUE,
        ] + $settingsForm;
      }
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $config = $this->config(self::CONFIG_ID);
    $ignoreConfig = ['form_build', 'form_token', 'op', 'form_build_id', 'form_id'];
    foreach ($form_state->getValues() as $key => $value) {
      if (!in_array($key, $ignoreConfig)) {
        $config->set($key, $value);
      }
    }
    $config->save();

    $this->messenger()->addStatus($this->t('The configuration has been updated.'));
  }

}
