<?php

namespace Drupal\media_accessibility_enhancer\Plugin\MediaDistantProvider;

use Drupal\media\OEmbed\Resource;
use Drupal\media_accessibility_enhancer\Plugin\MediaDistantProviderBase;

/**
 * Defined DailymotionMediaDistantProvider class.
 *
 * @MediaDistantProvider(
 *   id = "dailymotion_media_distant_provider",
 *   label = @Translation("Dailymotion Media Distant Provider"),
 *   oembed_provider = "Dailymotion"
 * )
 */
class DailymotionMediaDistantProvider extends MediaDistantProviderBase {

  /**
   * {@inheritdoc}
   */
  public function getDuration(?Resource $resource, string $url): string {
    $videoId = $this->getDailymotionVideoId($url);
    return $this->fetchDailymotionDurationSeconds($videoId) ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getTranscription(?Resource $resource, string $url, string $language): string {
    $videoId = $this->getDailymotionVideoId($url);
    $transcriptions = $this->fetchDailymotionSubtitles($videoId);

    if (!empty($transcriptions['list'])) {
      foreach ($transcriptions['list'] as $transcription) {
        // @todo because we no found dailymotion video with subtitle without auto generation (because data isn't on api).
      }
    }
    else {
      return $this->t('No transcription found');
    }

    return '';
  }

  /**
   * Fetch Dailymotion duration.
   *
   * @param string $videoId
   *   Dailymotion video ID.
   *
   * @return int|null
   *   Duration in seconds or NULL on error.
   */
  public function fetchDailymotionDurationSeconds(string $videoId): ?int {
    try {
      $resultat = $this->httpClient->request('GET', 'https://api.dailymotion.com/video/' . $videoId, [
        'query' => [
          'fields' => 'duration',
        ],
        'timeout' => 8,
      ]);
      $data = json_decode($resultat->getBody(), TRUE);

      return isset($data['duration']) ? (int) $data['duration'] : NULL;
    }
    catch (\Throwable $e) {
      $this->logger->get('media_accessibility_enhancer')->error('Dailymotion duration fetch error: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Fetch Dailymotion Subtitles.
   *
   * @param string $videoId
   *   Dailymotion video ID.
   *
   * @return array|null
   *   Subtitle array or NULL on error.
   */
  public function fetchDailymotionSubtitles(string $videoId): ?array {
    try {
      $resultat = $this->httpClient->request('GET', 'https://api.dailymotion.com/video/' . $videoId . '/subtitles');
      return json_decode($resultat->getBody(), TRUE);
    }
    catch (\Throwable $e) {
      $this->logger->get('media_accessibility_enhancer')->error('Dailymotion transcription fetch error: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Fetch Dailymotion Subtitle.
   *
   * @param string $videoId
   *   Dailymotion video ID.
   * @param string $url
   *   Dailymotion url from subtitles list.
   *
   * @return int|null
   *   Duration in seconds or NULL on error.
   */
  public function fetchDailymotionSubtitle(string $videoId, string $url): ?int {
    try {
      $resultat = $this->httpClient->request('GET', $url);
      return json_decode($resultat->getBody(), TRUE);
    }
    catch (\Throwable $e) {
      $this->logger->error('Dailymotion transcription item fetch error: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Extract a Dailymotion video ID from common URL formats.
   *
   * @param string $url
   *   Video URL.
   *
   * @return string
   *   Video ID.
   */
  private function getDailymotionVideoId(string $url): string {
    if (preg_match('~dailymotion\.com/(?:video|hub)/([A-Za-z0-9]+)~i', $url, $matches)) {
      return $matches[1];
    }

    if (preg_match('~dai\.ly/([A-Za-z0-9]+)~i', $url, $matches)) {
      return $matches[1];
    }

    return '';
  }

}
