<?php

namespace Drupal\media_accessibility_enhancer\Plugin\MediaDistantProvider;

use Drupal\media\OEmbed\Resource;
use Drupal\media_accessibility_enhancer\Plugin\MediaDistantProviderBase;
use GuzzleHttp\Psr7\HttpFactory;
use MrMySQL\YoutubeTranscript\Exception\PoTokenRequiredException;
use MrMySQL\YoutubeTranscript\Exception\YouTubeRequestFailedException;
use MrMySQL\YoutubeTranscript\TranscriptList;
use MrMySQL\YoutubeTranscript\TranscriptListFetcher;

/**
 * Defined YoutubeMediaDistantProvider class.
 *
 * @MediaDistantProvider(
 *   id = "youtube_media_distant_provider",
 *   label = @Translation("Youtube Media Distant Provider"),
 *   oembed_provider = "YouTube"
 * )
 */
class YoutubeMediaDistantProvider extends MediaDistantProviderBase {

  /**
   * {@inheritdoc}
   */
  public function getDuration(?Resource $resource, string $url): string {
    $videoId = $this->getYouTubeVideoId($url);
    $data = $this->fetchYouTubeDataContentDetails($videoId);

    if (!empty($data)) {
      $iso = $data['items'][0]['contentDetails']['duration'] ?? NULL;
    }

    return isset($iso) ? $this->isoToSeconds($iso) : '';
  }

  /**
   * {@inheritdoc}
   */
  public function getTranscription(?Resource $resource, string $url, string $language): string {
    $videoId = $this->getYouTubeVideoId($url);
    $transcriptList = $this->fetchYouTubeDataCaptions($videoId);
    $transcripts = '';
    /** @var \MrMySQL\YoutubeTranscript\Transcript $caption */
    foreach ($transcriptList as $caption) {
      if ($caption->language_code === $language) {
        try {
          $data = $caption->fetch(TRUE);
          $transcripts = html_entity_decode(
            implode(' ', array_column($data, 'text')),
            ENT_QUOTES | ENT_HTML5,
            'UTF-8'
          );
        }
        catch (PoTokenRequiredException | YouTubeRequestFailedException $e) {
          $this->logger->get('media_accessibility_enhancer')->error($e->getMessage());
        }
      }
    }

    return $transcripts;
  }

  /**
   * {@inheritdoc}
   */
  public function defineSettings(?array $config): array {
    $form = [];

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('API KEY'),
      '#default_value' => $config['api_key'] ?? NULL,
    ];

    return $form;
  }

  /**
   * Get video id from url youtube.
   *
   * @param string $url
   *   This is current url.
   *
   * @return string|null
   *   Return video id.
   */
  public function getYouTubeVideoId(string $url): ?string {
    $parts = parse_url($url);

    if (!isset($parts['host'])) {
      return NULL;
    }

    if (strpos($parts['host'], 'youtu.be') !== FALSE) {
      return ltrim($parts['path'], '/');
    }

    if (strpos($parts['host'], 'youtube.com') !== FALSE) {
      parse_str($parts['query'] ?? '', $query);
      return $query['v'] ?? NULL;
    }

    return NULL;
  }

  /**
   * Convert ISO duration to seconds.
   *
   * @param string $iso
   *   ISO duration.
   *
   * @return int
   *   Seconds.
   */
  private function isoToSeconds(string $iso): int {
    try {
      $start = new \DateTimeImmutable('@0');
      $end = $start->add(new \DateInterval($iso));

      return $end->getTimestamp();
    }
    catch (\Throwable) {
      return 0;
    }
  }

  /**
   * Fetch YouTube content details.
   *
   * @param string $videoId
   *   YouTube video ID.
   *
   * @return array|null
   *   Duration in seconds or NULL on error.
   */
  public function fetchYouTubeDataContentDetails(string $videoId): ?array {
    $apiKey = $this->getApiKey();

    try {
      $res = $this->httpClient->request('GET', 'https://www.googleapis.com/youtube/v3/videos', [
        'query' => [
          'id' => $videoId,
          'part' => 'contentDetails',
          'key' => $apiKey,
          'fields' => 'items(contentDetails(duration))',
        ],
        'timeout' => 8,
      ]);

      return json_decode($res->getBody(), TRUE);
    }
    catch (\Throwable $e) {
      $this->logger->get('media_accessibility_enhancer')->error('YouTube fetch error: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Fetch YouTube captions.
   *
   * @param string $videoId
   *   YouTube video ID.
   *
   * @return \MrMySQL\YoutubeTranscript\TranscriptList
   *   Duration in seconds or NULL on error.
   */
  public function fetchYouTubeDataCaptions(string $videoId): TranscriptList {
    $requestFactory = new HttpFactory();
    $streamFactory = new HttpFactory();
    $fetcher = new TranscriptListFetcher($this->httpClient, $requestFactory, $streamFactory);
    return $fetcher->fetch($videoId);
  }

  /**
   * Get api key from settings.
   *
   * @return string|null
   *   Return api key.
   */
  public function getApiKey(): ?string {
    $videoApiSettings = $this->getSettings();
    if (empty($videoApiSettings['api_key'])) {
      $this->logger->get('media_accessibility_enhancer')->alert('YouTube API key is not set.');
      return NULL;
    }

    $apiKey = $this->keyRepository->getKey($videoApiSettings['api_key'])->getKeyValue();

    if (!$apiKey) {
      $this->logger->get('media_accessibility_enhancer')->alert('YouTube API key is not set.');
      return NULL;
    }

    return $apiKey;
  }

}
