<?php

namespace Drupal\media_accessibility_enhancer\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\key\KeyRepositoryInterface;
use Drupal\media\OEmbed\Resource;
use Drupal\media\OEmbed\ResourceFetcherInterface;
use Drupal\media_accessibility_enhancer\Form\MediaDistantProviderSettingsForm;
use GuzzleHttp\Client;
use Psr\Http\Client\ClientExceptionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defined MediaDistantProviderBase class.
 */
abstract class MediaDistantProviderBase extends PluginBase implements MediaDistantProviderInterface, ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    $plugin_definition,
    protected Client $httpClient,
    protected ConfigFactory $configFactory,
    protected KeyRepositoryInterface $keyRepository,
    protected ResourceFetcherInterface $resourceFetcher,
    protected LoggerChannelFactoryInterface $logger,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('http_client'),
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('media.oembed.resource_fetcher'),
      $container->get('logger.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getTranscription(?Resource $resource, string $url, string $language): string {
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function getDuration(?Resource $resource, string $url): string {
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function getResourceInformation(string $url): ?Resource {
    $resource_url = $this->discoverResourceUrl($url);
    if ($resource_url) {
      return $this->resourceFetcher->fetchResource($resource_url);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function defineSettings(?array $config): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getSettings(): array {
    return $this->configFactory->get(MediaDistantProviderSettingsForm::CONFIG_ID)->get($this->getPluginId()) ?? [];
  }

  /**
   * Runs oEmbed discovery and returns the endpoint URL if successful.
   *
   * @param string $url
   *   The resource's URL.
   *
   * @return string|bool
   *   URL of the oEmbed endpoint, or FALSE if the discovery was unsuccessful.
   */
  protected function discoverResourceUrl($url) {
    try {
      $response = $this->httpClient->get($url);
    }
    catch (ClientExceptionInterface) {
      return FALSE;
    }

    $document = Html::load($response->getBody()->getContents());
    $xpath = new \DOMXPath($document);

    return $this->findUrl($xpath, 'json') ?: $this->findUrl($xpath, 'xml');
  }

  /**
   * Tries to find the oEmbed URL in a DOM.
   *
   * @param \DOMXPath $xpath
   *   Page HTML as DOMXPath.
   * @param string $format
   *   Format of oEmbed resource. Possible values are 'json' and 'xml'.
   *
   * @return bool|string
   *   A URL to an oEmbed resource or FALSE if not found.
   */
  protected function findUrl(\DOMXPath $xpath, $format) {
    $result = $xpath->query("//link[@type='application/$format+oembed']");
    return $result->length ? $result->item(0)->getAttribute('href') : FALSE;
  }

}
