<?php

declare(strict_types=1);

namespace Drupal\media_avportal\StreamWrapper;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Stream wrapper for the remote AV Portal photos.
 *
 * @codingStandardsIgnoreStart PSR1.Methods.CamelCapsMethodName
 */
class AvPortalPhotoStreamWrapper extends AvPortalStreamWrapper {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function getName(): string|TranslatableMarkup {
    return $this->t('AV Portal photo stream wrapper');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string|TranslatableMarkup {
    return $this->t('Stream wrapper for the remote location where AV Portal photos can be found.');
  }

  /**
   * {@inheritdoc}
   */
  public function getExternalUrl(): ?string {
    $resource = $this->avPortalClient->getResource($this->getPath());
    if (!$resource) {
      return NULL;
    }

    return $this->configuration->get('photos_base_uri') . $resource->getPhotoUri();
  }

  /**
   * {@inheritdoc}
   */
  public function stream_open($path, $mode, $options, &$opened_path): bool {
    if (!in_array($mode, ['r', 'rb'])) {
      return FALSE;
    }

    $this->uri = $path;
    $url = $this->getExternalUrl();
    if (!$url) {
      return FALSE;
    }

    $url = $this->getFullExternalUrl($url);

    return parent::stream_open($url, $mode, $options, $opened_path);
  }

  /**
   * {@inheritdoc}
   */
  public function url_stat($path, $flags): array|false {
    $this->uri = $path;

    $resource = $this->avPortalClient->getResource($this->getPath());
    if (!$resource) {
      return FALSE;
    }

    $url = $this->configuration->get('photos_base_uri') . $resource->getPhotoUri();
    $url = $this->getFullExternalUrl($url);

    return parent::url_stat($url, $flags);
  }

  /**
   * {@inheritdoc}
   */
  public function unlink($path): bool {
    return TRUE;
  }

  /**
   * Extract resource path from target.
   *
   * @return string
   *   The resource path.
   */
   protected function getPath(): string {
      $path = str_replace('\\', '/', $this->getTarget());
      // The stream wrapper URI expects an image extension because otherwise it
      // cannot be used for generating image styles. When the latter happens, the
      // extension is checked to determine whether the file is supported by the
      // available image toolkit. We default to (assume) JPG as the resources
      // are photos.
      // @see \Drupal\image\Entity\ImageStyle::supportsUri().
      return str_replace('.jpg', '', $path);
   }

}
