<?php

declare(strict_types=1);

namespace Drupal\Tests\media_avportal\src\Unit;

use Drupal\media_avportal\AvPortalResource;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the AvPortalResource class.
 *
 * @coversDefaultClass \Drupal\media_avportal\AvPortalResource
 * @group media_avportal
 */
class AvPortalResourceTest extends UnitTestCase {

  /**
   * Tests getPhotoThumbnailUrl() method.
   *
   * @param array $data
   *   The photo resource with all resolutions.
   *
   * @dataProvider photoThumbnailsResourceDataProvider
   * @covers ::getPhotoThumbnailUrl
   */
  public function testGetPhotoThumbnailUrl(array $data): void {
    $resource = new AvPortalResource($data);
    // Assert we get the medium resolution when all resolution are present.
    $this->assertEquals('medium.jpg', $resource->getThumbnailUrl());

    // Remove medium, assert for low resolution.
    unset($data['media_json']['MED']);
    $resource = new AvPortalResource($data);
    $this->assertEquals('low.jpg', $resource->getThumbnailUrl());

    // Remove low, assert for high resolution.
    unset($data['media_json']['LOW']);
    $resource = new AvPortalResource($data);
    $this->assertEquals('high.jpg', $resource->getThumbnailUrl());

    // Unset all resolutions.
    unset($data['media_json']['HIGH']);
    $resource = new AvPortalResource($data);
    $this->assertNull($resource->getThumbnailUrl());
  }

  /**
   * Tests getTitle() method.
   *
   * @param array $data
   *   The AV Portal resource with all possible titles.
   * @param string|null $langcode
   *   The language in which to return the title.
   * @param string|null $expected
   *   The expected value.
   *
   * @dataProvider titleResourceDataProvider
   * @covers ::getTitle
   */
  public function testGetTitle(array $data, ?string $langcode, ?string $expected): void {
    $resource = new AvPortalResource($data);

    // Pass the language parameter only if it has been specified.
    $title = $langcode === NULL ? $resource->getTitle() : $resource->getTitle($langcode);
    $this->assertSame($expected, $title);
  }

  /**
   * Tests getCaption() method.
   *
   * @param array $data
   *   The AV Portal resource with all possible captions.
   * @param string|null $langcode
   *   The language in which to return the caption.
   * @param string|null $expected
   *   The expected value.
   *
   * @dataProvider captionResourceDataProvider
   * @covers ::getCaption
   */
  public function testGetCaption(array $data, ?string $langcode, ?string $expected): void {
    $resource = new AvPortalResource($data);

    // Pass the language parameter only if it has been specified.
    $title = $langcode === NULL ? $resource->getCaption() : $resource->getCaption($langcode);
    $this->assertSame($expected, $title);
  }

  /**
   * Provide photo resources with PHOTO and REPORTAGE types.
   *
   * @return array
   *   List of photo resources.
   */
  public static function photoThumbnailsResourceDataProvider(): array {
    return [
      [
        [
          'ref' => 'P-038924/00-15',
          'type' => 'PHOTO',
          'media_json' =>
            [
              'MED' =>
                [
                  'PIXH' => 426,
                  'PIXL' => 640,
                  'PATH' => 'medium.jpg',
                ],
              'HIGH' =>
                [
                  'PIXH' => 3455,
                  'PIXL' => 5183,
                  'PATH' => 'high.jpg',
                ],
              'LOW' =>
                [
                  'PIXH' => 133,
                  'PIXL' => 200,
                  'PATH' => 'low.jpg',
                ],
            ],
        ],
      ],
      [
        [
          'ref' => 'P-038924/00-15',
          'type' => 'REPORTAGE',
          'media_json' =>
            [
              'MED' =>
                [
                  'PIXH' => 426,
                  'PIXL' => 640,
                  'PATH' => 'medium.jpg',
                ],
              'HIGH' =>
                [
                  'PIXH' => 3455,
                  'PIXL' => 5183,
                  'PATH' => 'high.jpg',
                ],
              'LOW' =>
                [
                  'PIXH' => 133,
                  'PIXL' => 200,
                  'PATH' => 'low.jpg',
                ],
            ],
        ],
      ],
    ];
  }

  /**
   * Data provider for getTitle() test method.
   *
   * @return array
   *   Test data and expectations.
   */
  public static function titleResourceDataProvider(): array {
    return [
      'missing titles_json' => [
        [
          'ref' => 'P-038924/00-15',
        ],
        NULL,
        NULL,
      ],
      'non-array titles_json' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => 'invalid title',
        ],
        NULL,
        NULL,
      ],
      'empty titles_json' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [],
        ],
        NULL,
        NULL,
      ],
      'titles_json with NULL values' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'FR' => NULL,
            'EN' => NULL,
          ],
        ],
        'FR',
        NULL,
      ],
      // @see https://www.php.net/manual/en/language.types.boolean.php#language.types.boolean.casting
      'title evaluable to FALSE on casting' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'IT' => '0',
          ],
        ],
        'IT',
        '0',
      ],
      'boolean title' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'IT' => FALSE,
          ],
        ],
        'IT',
        NULL,
      ],
      'scalar title' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'EN' => ['English title.'],
          ],
        ],
        'EN',
        NULL,
      ],
      // English is the default langcode used when none is passed.
      'no langcode specified / existing title' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'FR' => 'French title.',
            'EN' => 'English title.',
          ],
        ],
        NULL,
        'English title.',
      ],
      'langcode specified / existing title' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'EN' => 'English title.',
            'FR' => 'French title.',
          ],
        ],
        'FR',
        'French title.',
      ],
      // English is the fallback langcode used when the requested is not found.
      'langcode specified / not existing title' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'IT' => 'Italian title.',
            'EN' => 'English title.',
          ],
        ],
        'FR',
        'English title.',
      ],
      // The first title available is returned when the fallback is not found.
      'no langcode specified / fallback title not existing' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'IT' => 'Italian title.',
            'FR' => 'French title.',
          ],
        ],
        NULL,
        'Italian title.',
      ],
      // Another iteration of the above with a different titles order.
      'no langcode specified / different order of titles / fallback title not existing' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'FR' => 'French title.',
            'IT' => 'Italian title.',
          ],
        ],
        NULL,
        'French title.',
      ],
      // When the requested language and the fallback one are not found, the
      // first available title is returned.
      'langcode specified / title and fallback title not existing' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'IT' => 'Italian title.',
            'DE' => 'German title.',
          ],
        ],
        'FR',
        'Italian title.',
      ],
      // Another iteration of the above with a different titles order.
      'langcode specified / different order of titles / title and fallback title not existing' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'DE' => 'German title.',
            'IT' => 'Italian title.',
          ],
        ],
        'FR',
        'German title.',
      ],
      'title with encoded characters and markup / langcode specified' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'FR' => 'French title <br />&lt;strong&gt;with&lt;/strong&gt; markup, encoded &#39;characters&#39; &amp; letters čö&įię.',
          ],
        ],
        'FR',
        'French title with markup, encoded \'characters\' & letters čö&įię.',
      ],
      'title with encoded characters and markup / fallback title' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'EN' => 'English title <br />&lt;strong&gt;with&lt;/strong&gt; markup, encoded &#39;characters&#39; &amp; letters čö&įię.',
          ],
        ],
        'FR',
        'English title with markup, encoded \'characters\' & letters čö&įię.',
      ],
      'title with encoded characters and markup / first available title' => [
        [
          'ref' => 'P-038924/00-15',
          'titles_json' => [
            'IT' => 'Italian title <br />&lt;strong&gt;with&lt;/strong&gt; markup, encoded &#39;characters&#39; &amp; letters čö&įię.',
          ],
        ],
        'FR',
        'Italian title with markup, encoded \'characters\' & letters čö&įię.',
      ],
      'title with more than 255 characters / html, encoded and multibyte characters' => [
        [
          'ref' => 'P-047441/00-05',
          'titles_json' => [
            'FR' => 'Conférence de presse de Margrethe Vestager, vice-présidente exécutive de la Commission européenne, sur un cas de pratique anti-concurrentielle :<br /> la Commission a infligé des amendes à Teva et à Cephalon pour avoir retardé l&#39;entrée sur le marché d&#39;un médicament générique moins cher',
          ],
        ],
        NULL,
        'Conférence de presse de Margrethe Vestager, vice-présidente exécutive de la Commission européenne, sur un cas de pratique anti-concurrentielle : la Commission a infligé des amendes à Teva et à Cephalon pour avoir retardé l\'entrée sur le marché d\'un…',
      ],
      'title with exactly 255 characters / html, encoded and multibyte characters' => [
        [
          'ref' => 'P-047441/00-05',
          'titles_json' => [
            'FR' => 'Conférence de presse de Margrethe Vestager, vice-présidente exécutive de la Commission européenne, sur un cas de pratique anti-concurrentielle :<br /> la Commission a infligé des amendes à Teva et à Cephalon pour avoir retardé l&#39;entrée sur le marché d&#39;un to255',
          ],
        ],
        NULL,
        'Conférence de presse de Margrethe Vestager, vice-présidente exécutive de la Commission européenne, sur un cas de pratique anti-concurrentielle : la Commission a infligé des amendes à Teva et à Cephalon pour avoir retardé l\'entrée sur le marché d\'un to255',
      ],
    ];
  }

  /**
   * Data provider for getCaption() test method.
   *
   * @return array
   *   Test data and expectations.
   */
  public static function captionResourceDataProvider(): array {
    return [
      // Data for not supported resource type.
      'not supported resource type' => [
        [
          'type' => 'VIDEO',
          'ref' => 'P-038924/00-15',
        ],
        NULL,
        NULL,
      ],

      // Data for Photo type.
      'missing summary_json' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
        ],
        NULL,
        NULL,
      ],
      'non-array summary_json' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'legend_json' => 'invalid title',
        ],
        NULL,
        NULL,
      ],
      'empty summary_json' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [],
        ],
        NULL,
        NULL,
      ],
      'summary_json with NULL values' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'FR' => NULL,
            'EN' => NULL,
          ],
        ],
        'FR',
        NULL,
      ],
      // @see https://www.php.net/manual/en/language.types.boolean.php#language.types.boolean.casting
      'summary_json evaluable to FALSE on casting' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'IT' => '0',
          ],
        ],
        'IT',
        '0',
      ],
      'summary_json boolean caption' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'IT' => FALSE,
          ],
        ],
        'IT',
        NULL,
      ],
      'summary_json scalar caption' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'EN' => ['English caption.'],
          ],
        ],
        'EN',
        NULL,
      ],
      // English is the default langcode used when none is passed.
      'no langcode specified for summary_json / existing caption' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'FR' => 'French caption.',
            'EN' => 'English caption.',
          ],
        ],
        NULL,
        'English caption.',
      ],
      'langcode specified for summary_json / existing caption' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'EN' => 'English title.',
            'FR' => 'French title.',
          ],
        ],
        'FR',
        'French title.',
      ],
      // English is the fallback langcode used when the requested is not found.
      'langcode specified for summary_json / not existing caption' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'IT' => 'Italian caption.',
            'EN' => 'English caption.',
          ],
        ],
        'FR',
        'English caption.',
      ],
      // The first title available is returned when the fallback is not found.
      'no langcode specified for summary_json / fallback caption not existing' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'IT' => 'Italian caption.',
            'FR' => 'French caption.',
          ],
        ],
        NULL,
        'Italian caption.',
      ],
      // Another iteration of the above with a different titles order.
      'no langcode specified in summary_json / different order of caption / fallback caption not existing' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'FR' => 'French caption.',
            'IT' => 'Italian caption.',
          ],
        ],
        NULL,
        'French caption.',
      ],
      // When the requested language and the fallback one are not found, the
      // first available caption is returned.
      'langcode specified for summary_json / caption and fallback caption not existing' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'IT' => 'Italian title.',
            'DE' => 'German title.',
          ],
        ],
        'FR',
        'Italian title.',
      ],
      // Another iteration of the above with a different caption order.
      'langcode specified / different order of captions in summary_json / caption and fallback caption not existing' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'DE' => 'German caption.',
            'IT' => 'Italian caption.',
          ],
        ],
        'FR',
        'German caption.',
      ],
      'caption with encoded characters and markup in summary_json / langcode specified' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'FR' => 'French caption <br />&lt;strong&gt;with&lt;/strong&gt; markup, encoded &#39;characters&#39; &amp; letters čö&įię.',
          ],
        ],
        'FR',
        'French caption with markup, encoded \'characters\' & letters čö&įię.',
      ],
      'caption with encoded characters and markup in summary_json / fallback caption' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'EN' => 'English capture <br />&lt;strong&gt;with&lt;/strong&gt; markup, encoded &#39;characters&#39; &amp; letters čö&įię.',
          ],
        ],
        'FR',
        'English capture with markup, encoded \'characters\' & letters čö&įię.',
      ],
      'caption with encoded characters and markup in summary_json / first available caption' => [
        [
          'type' => 'PHOTO',
          'ref' => 'P-038924/00-15',
          'summary_json' => [
            'IT' => 'Italian capture <br />&lt;strong&gt;with&lt;/strong&gt; markup, encoded &#39;characters&#39; &amp; letters čö&įię.',
          ],
        ],
        'FR',
        'Italian capture with markup, encoded \'characters\' & letters čö&įię.',
      ],

      // Data for Reportage type.
      'missing legend_json' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
        ],
        NULL,
        NULL,
      ],
      'non-array legend_json' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => 'invalid title',
        ],
        NULL,
        NULL,
      ],
      'empty legend_json' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [],
        ],
        NULL,
        NULL,
      ],
      'legend_json with NULL values' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'FR' => NULL,
            'EN' => NULL,
          ],
        ],
        'FR',
        NULL,
      ],
      // @see https://www.php.net/manual/en/language.types.boolean.php#language.types.boolean.casting
      'caption evaluable to FALSE on casting' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'IT' => '0',
          ],
        ],
        'IT',
        '0',
      ],
      'boolean caption' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'IT' => FALSE,
          ],
        ],
        'IT',
        NULL,
      ],
      'scalar caption' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'EN' => ['English caption.'],
          ],
        ],
        'EN',
        NULL,
      ],
      // English is the default langcode used when none is passed.
      'no langcode specified / existing caption' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'FR' => 'French caption.',
            'EN' => 'English caption.',
          ],
        ],
        NULL,
        'English caption.',
      ],
      'langcode specified / existing caption' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'EN' => 'English title.',
            'FR' => 'French title.',
          ],
        ],
        'FR',
        'French title.',
      ],
      // English is the fallback langcode used when the requested is not found.
      'langcode specified / not existing caption' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'IT' => 'Italian caption.',
            'EN' => 'English caption.',
          ],
        ],
        'FR',
        'English caption.',
      ],
      // The first title available is returned when the fallback is not found.
      'no langcode specified / fallback caption not existing' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'IT' => 'Italian caption.',
            'FR' => 'French caption.',
          ],
        ],
        NULL,
        'Italian caption.',
      ],
      // Another iteration of the above with a different titles order.
      'no langcode specified / different order of caption / fallback caption not existing' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'FR' => 'French caption.',
            'IT' => 'Italian caption.',
          ],
        ],
        NULL,
        'French caption.',
      ],
      // When the requested language and the fallback one are not found, the
      // first available caption is returned.
      'langcode specified / caption and fallback caption not existing' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'IT' => 'Italian title.',
            'DE' => 'German title.',
          ],
        ],
        'FR',
        'Italian title.',
      ],
      // Another iteration of the above with a different caption order.
      'langcode specified / different order of captions / caption and fallback caption not existing' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'DE' => 'German caption.',
            'IT' => 'Italian caption.',
          ],
        ],
        'FR',
        'German caption.',
      ],
      'caption with encoded characters and markup / langcode specified' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'FR' => 'French caption <br />&lt;strong&gt;with&lt;/strong&gt; markup, encoded &#39;characters&#39; &amp; letters čö&įię.',
          ],
        ],
        'FR',
        'French caption with markup, encoded \'characters\' & letters čö&įię.',
      ],
      'caption with encoded characters and markup / fallback caption' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'EN' => 'English capture <br />&lt;strong&gt;with&lt;/strong&gt; markup, encoded &#39;characters&#39; &amp; letters čö&įię.',
          ],
        ],
        'FR',
        'English capture with markup, encoded \'characters\' & letters čö&įię.',
      ],
      'caption with encoded characters and markup / first available caption' => [
        [
          'type' => 'REPORTAGE',
          'ref' => 'P-038924/00-15',
          'legend_json' => [
            'IT' => 'Italian capture <br />&lt;strong&gt;with&lt;/strong&gt; markup, encoded &#39;characters&#39; &amp; letters čö&įię.',
          ],
        ],
        'FR',
        'Italian capture with markup, encoded \'characters\' & letters čö&įię.',
      ],
    ];
  }

}
