<?php

declare(strict_types=1);

namespace Drupal\Tests\media_duplicate_check\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\media\Entity\Media;
use Drupal\file\Entity\File;

/**
 * Functional tests for Media Duplicate Check module.
 *
 * @group media_duplicate_check
 */
class MediaDuplicateCheckTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'media_duplicate_check',
    'media',
    'file',
    'image',
    'field_ui',
  ];

  /**
   * A user with permission to administer site configuration.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create an admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'create media',
      'edit own media',
      'access media overview',
    ]);
  }

  /**
   * Tests the module configuration form.
   */
  public function testConfigurationForm(): void {
    $this->drupalLogin($this->adminUser);

    // Visit the configuration page.
    $this->drupalGet('/admin/config/media/duplicate-check');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Enable duplicate checking');

    // Test form submission.
    $edit = [
      'enabled' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Tests that the configuration affects the duplicate checker service.
   */
  public function testServiceConfiguration(): void {
    $config = $this->config('media_duplicate_check.settings');
    $config->set('enabled', TRUE);
    $config->set('media_types', ['image']);
    $config->save();

    /** @var \Drupal\media_duplicate_check\Service\MediaDuplicateChecker $service */
    $service = \Drupal::service('media_duplicate_check.duplicate_checker');

    // Test that the service exists and is properly configured.
    $this->assertNotNull($service);

    // Test finding duplicates with empty result.
    $result = $service->findDuplicatesByFilename('nonexistent.jpg', 'image');
    $this->assertNotNull($result);
    $this->assertCount(0, $result);
  }

  /**
   * Tests permissions and access control.
   */
  public function testPermissions(): void {
    // Test anonymous user cannot access configuration.
    $this->drupalGet('/admin/config/media/duplicate-check');
    $this->assertSession()->statusCodeEquals(403);

    // Test admin user can access configuration.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/media/duplicate-check');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests module installation and default configuration.
   */
  public function testModuleInstallation(): void {
    // Check that default configuration is created.
    $config = $this->config('media_duplicate_check.settings');
    $this->assertNotNull($config->get('enabled'));

    // Check that the service is available.
    $service = \Drupal::service('media_duplicate_check.duplicate_checker');
    $this->assertNotNull($service);
  }

}