<?php

declare(strict_types=1);

namespace Drupal\media_duplicate_check\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Media Duplicate Check settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'media_duplicate_check_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['media_duplicate_check.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('media_duplicate_check.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable duplicate checking'),
      '#description' => $this->t('When enabled, the system will check for duplicate filenames when uploading media.'),
      '#default_value' => $config->get('enabled') ?? TRUE,
    ];


    $form['check_media_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Check these media types'),
      '#description' => $this->t('Select which media types should be checked for duplicates.'),
      '#options' => [
        'image' => $this->t('Image'),
        'document' => $this->t('Document'),
        'svg_image' => $this->t('SVG Image'),
        'video' => $this->t('Video'),
        'audio' => $this->t('Audio'),
        'remote_video' => $this->t('Remote Video'),
      ],
      '#default_value' => $config->get('check_media_types') ?? ['image', 'document', 'svg_image'],
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['check_by_hash'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Check by file content (MD5 hash)'),
      '#description' => $this->t('If checked, files will be compared by their content (MD5 hash) to detect true duplicates even if filenames differ.'),
      '#default_value' => $config->get('check_by_hash') ?? TRUE,
    ];
    
    $form['advanced']['check_filename_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Check filename only'),
      '#description' => $this->t('If checked, only the filename will be compared. If unchecked, file hash will also be considered.'),
      '#default_value' => $config->get('check_filename_only') ?? FALSE,
      '#states' => [
        'visible' => [
          ':input[name="check_by_hash"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['advanced']['case_sensitive'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Case-sensitive comparison'),
      '#description' => $this->t('If checked, filename comparison will be case-sensitive.'),
      '#default_value' => $config->get('case_sensitive') ?? FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('media_duplicate_check.settings')
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('check_media_types', array_filter($form_state->getValue('check_media_types')))
      ->set('check_by_hash', $form_state->getValue('check_by_hash'))
      ->set('check_filename_only', $form_state->getValue('check_filename_only'))
      ->set('case_sensitive', $form_state->getValue('case_sensitive'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
