<?php

declare(strict_types=1);

namespace Drupal\media_duplicate_check\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for handling duplicate upload confirmations.
 */
class ConfirmController extends ControllerBase {

  /**
   * Handle confirmation AJAX request.
   */
  public function confirm(Request $request) {
    $filename = $request->request->get('filename');
    $confirmed = $request->request->get('confirmed');
    
    if ($filename && $confirmed === '1') {
      // Store confirmation in session using the filename as key
      $session = $request->getSession();
      $session->set('media_duplicate_confirmed_' . md5($filename), TRUE);
      
      // Also try to extract base filename and store that too for compatibility
      $duplicate_checker = \Drupal::service('media_duplicate_check.duplicate_checker');
      $base_filename = $duplicate_checker->getBaseFilename($filename);
      $extension = pathinfo($filename, PATHINFO_EXTENSION);
      $full_base_filename = $base_filename . '.' . $extension;
      $session->set('media_duplicate_confirmed_' . md5($full_base_filename), TRUE);
      
      return new JsonResponse(['status' => 'success']);
    }
    
    return new JsonResponse(['status' => 'error'], 400);
  }

  /**
   * Check for duplicates using original filename (before upload).
   */
  public function checkOriginal(Request $request) {
    $filename = $request->request->get('filename');
    
    if (!$filename) {
      return new JsonResponse(['status' => 'error', 'message' => 'No filename provided'], 400);
    }
    
    // Get the duplicate checker service
    $duplicate_checker = \Drupal::service('media_duplicate_check.duplicate_checker');
    
    // Check for duplicates using the original filename
    $duplicates = $duplicate_checker->findDuplicatesByOriginalFilename($filename);
    
    if (!empty($duplicates)) {
      // Prepare duplicate data for response
      $duplicate_data = [];
      foreach ($duplicates as $media) {
        $preview_data = $duplicate_checker->getMediaPreviewData($media);
        $duplicate_data[] = [
          'id' => $preview_data['id'],
          'name' => $preview_data['name'],
          'type' => $preview_data['type'],
          'created_formatted' => $preview_data['created_formatted'],
          'url' => $preview_data['url'],
        ];
      }
      
      return new JsonResponse([
        'status' => 'duplicates_found',
        'duplicates' => $duplicate_data,
        'count' => count($duplicates),
      ]);
    }
    
    return new JsonResponse([
      'status' => 'no_duplicates',
      'duplicates' => [],
    ]);
  }
}
