<?php

namespace Drupal\media_entity_calameo;

use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Utility\Error;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

/**
 * Get Calameo's publication information.
 */
class CalameoManager implements CalameoManagerInterface {

  const GENERIC_CALAMEO_API = 'https://api.calameo.com/1.0';
  const UPLOAD_CALAMEO_API = 'https://upload.calameo.com/1.0';

  /**
   * Calameo manager constructor.
   *
   * @param \GuzzleHttp\Client $httpClient
   *   A Guzzle HTTP Client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   A logger factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory service.
   */
  public function __construct(protected Client $httpClient, protected LoggerChannelFactoryInterface $loggerFactory, protected ConfigFactoryInterface $configFactory) {}

  /**
   * {@inheritdoc}
   */
  public function getPublicationInfos($id) {
    if (!$this->hasApiSettings()) {
      $this->loggerFactory->get('media_entity_calameo')->error(
        "Api key or Api secret is missing"
      );
      return FALSE;
    }

    if (filter_var($id, FILTER_VALIDATE_URL)) {
      $this->loggerFactory->get('media_entity_calameo')->error(
        "Book ID cannot be retrieved from publication URL."
      );
      return FALSE;
    }

    /** @var \Drupal\Core\Config\ImmutableConfig $settings */
    $settings = $this->configFactory->get('media_entity_calameo.settings');
    $api_key = $settings->get('api_key');
    $secret_key = $settings->get('api_secret');

    $options = [
      'apikey' => $api_key,
      'action' => 'API.getBookInfos',
      'book_id' => $id,
      'output' => 'JSON',
    ];

    $signature = $secret_key;
    $signature .= 'actionAPI.getBookInfos';
    $signature .= "apikey{$api_key}";
    $signature .= "book_id{$id}";
    $signature .= "outputJSON";
    $options['signature'] = md5($signature);

    try {
      $response = $this->httpClient->request(
        'GET',
        self::GENERIC_CALAMEO_API . '?' . UrlHelper::buildQuery($options),
        ['timeout' => 5]
      );
      if ($response->getStatusCode() === 200) {
        $data = Json::decode($response->getBody()->getContents());
      }
    }
    catch (RequestException $e) {
      $this->loggerFactory->get('media_entity_calameo')->error(
        "Could not retrieve Calameo publication information: $id.",
        Error::decodeException($e)
      );
    }

    return $data ?? FALSE;
  }

  /**
   * To know if API settings are filling.
   *
   * @return bool
   *   True if settings are filling or false if not.
   */
  protected function hasApiSettings(): bool {
    $settings = $this->configFactory->get('media_entity_calameo.settings');
    return $settings->get('api_key') && $settings->get('api_secret');
  }

}
