<?php

namespace Drupal\media_entity_calameo\Plugin\Field\FieldFormatter;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\media_entity_calameo\Plugin\media\Source\Calameo;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'calameo_embed' formatter.
 *
 * @FieldFormatter(
 *   id = "calameo_embed",
 *   label = @Translation("Calameo embed"),
 *   field_types = {
 *     "string"
 *   }
 * )
 */
class CalameoEmbedFormatter extends FormatterBase {

  /**
   * Constructs an CalameoEmbedFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin ID for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, protected ConfigFactoryInterface $configFactory) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    /** @var \Drupal\media\MediaInterface $media */
    $media = $items->getEntity();
    $source = $media->getSource();
    if (!$source instanceof Calameo) {
      return [];
    }

    $element = [];
    $publication_id = $source->getMetadata($media, 'ID');
    foreach ($items as $delta => $item) {
      $source_value = $item->value;
      if (!$source_value) {
        break;
      }

      $options = [
        'query' => [
          'mode' => $this->getSetting('mode'),
          'view' => $this->getSetting('view'),
        ],
      ];
      $uri = $source_value;
      if (stripos($uri, 'http') === FALSE || $this->getSettingFromForm('force_domain')) {
        $uri = "//v.calameo.com/";
        $options['query']['bkcode'] = $publication_id;
      }

      $options['query'] = array_filter($options['query']);
      $iframe_url = Url::fromUri($uri, $options);

      $element[$delta] = [
        '#theme' => 'media_calameo',
        '#url' => $iframe_url->toString(),
        '#width' => $this->getSetting('width'),
        '#height' => $this->getSetting('height'),
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    return $field_definition->getTargetEntityTypeId() === 'media';
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'mode' => '',
      'view' => '',
      'width' => '480',
      'height' => '640',
    ] + parent::defaultSettings();
  }

  /**
   * Returns the value of a setting, or from Calameo administration config form.
   *
   * @param string $key
   *   The setting name.
   *
   * @return string
   *   The setting value.
   */
  public function getSettingFromForm(string $key): string {
    /** @var \Drupal\Core\Config\ImmutableConfig $settings */
    $settings = $this->configFactory->get('media_entity_calameo.settings');
    return $this->getSetting($key) ?: ($settings->get($key) ?? '');
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Display mode'),
      '#options' => [
        '' => $this->t('Automatic'),
        'viewer' => $this->t('Viewer'),
        'mini' => $this->t('Mini'),
      ],
      '#default_value' => $this->getSettingFromForm('mode'),
      '#description' => $this->t('Embed display mode.'),
    ];

    $elements['view'] = [
      '#type' => 'select',
      '#title' => $this->t('Display view'),
      '#options' => [
        '' => $this->t('Automatic'),
        'book' => $this->t('Book'),
        'slide' => $this->t('Slide'),
        'scroll' => $this->t('Scroll'),
      ],
      '#default_value' => $this->getSettingFromForm('view'),
      '#description' => $this->t('Embed display view.'),
    ];

    $elements['width'] = [
      '#type' => 'number',
      '#title' => $this->t('Width'),
      '#default_value' => $this->getSetting('width'),
      '#min' => 1,
      '#description' => $this->t('Width of Calameo embed.'),
    ];

    $elements['height'] = [
      '#type' => 'number',
      '#title' => $this->t('Height'),
      '#default_value' => $this->getSetting('height'),
      '#min' => 1,
      '#description' => $this->t('Height of Calameo embed.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    return [
      $this->t('Embed display mode: @mode', [
        '@mode' => $this->getSettingFromForm('mode') ?: $this->t('Automatic'),
      ]),
      $this->t('Embed display view: @view', [
        '@view' => $this->getSettingFromForm('view') ?: $this->t('Automatic'),
      ]),
      $this->t('Width: @width px', [
        '@width' => $this->getSetting('width'),
      ]),
      $this->t('Height: @height px', [
        '@height' => $this->getSetting('height'),
      ]),
    ];
  }

}
