<?php

namespace Drupal\Tests\media_gallery\Kernel;

use Drupal\media_gallery\Entity\MediaGallery;
use Drupal\media\Entity\Media;
use Drupal\Core\Render\Element;

use Drupal\Tests\media_gallery\Kernel\Traits\MediaGalleryPagerTrait;

/**
 * Tests the reverse pagination logic in media_gallery.module.
 *
 * @group media_gallery
 */
class MediaGalleryReversePaginationTest extends MediaGalleryKernelTestBase {
  use MediaGalleryPagerTrait;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    // The media_gallery.module file is not loaded by default in kernel tests,
    // but it's where the function we want to test lives.
    $this->container->get('module_handler')->load('media_gallery');
  }

  /**
   * Tests that reversing a gallery with a pager works correctly.
   *
   * This test is expected to fail with the buggy implementation.
   */
  public function testReverseWithPager() {
    $total_items = 15;
    $items_per_page = 12;

    // Given a media gallery with $total_items media items, with pager enabled
    // and reverse order set.
    $gallery = MediaGallery::create([
      'title' => 'My Test Gallery',
      'use_pager' => TRUE,
      'reverse' => TRUE,
    ]);

    $media_items = [];
    for ($i = 0; $i < $total_items; $i++) {
      $media = Media::create([
        'bundle' => 'image',
        'name' => "Image $i",
      ]);
      $media->save();
      $media_items[] = ['target_id' => $media->id()];
    }
    $gallery->set('images', $media_items);
    $gallery->save();

    // Manually construct the render array for the gallery, mimicking what
    // the field formatter would produce during a full render.
    $elements = [
      '#media_gallery' => $gallery,
      '#theme' => 'media_gallery',
      'images' => [
        '#items' => $gallery->get('images'),
        '#formatter' => 'photoswipe_field_formatter',
        '#object' => $gallery,
      ],
    ];

    // Populate the individual image render arrays within $elements['images'].
    foreach ($gallery->get('images') as $delta => $item) {
      $media = $item->entity;
      $elements['images'][$delta] = [
      // Simple markup for testing.
        '#markup' => 'Image ' . $delta,
      // Add the index for tracking.
        '#item_index' => $delta,
      // Attach the media entity for alter functions.
        '#media' => $media,
      ];
    }

    $variables = [
      'elements' => $elements,
      'content' => [
        'images' => $elements['images'],
      ],
    ];

    // When processing the first page.
    $this->givenTheCurrentPageIs(0);
    $page1_vars = $variables;
    media_gallery_preprocess_media_gallery($page1_vars);

    // Then expect page 1 to have the last $items_per_page items in reverse
    // order.
    $expected_page1_count = $items_per_page;
    $expected_page1_first_index = $total_items - 1;
    $expected_page1_last_index = $total_items - $items_per_page;

    $this->assertCount($expected_page1_count, Element::children($page1_vars['images']));
    $this->assertEquals($expected_page1_first_index, $page1_vars['images'][0]['#item_index']);
    $this->assertEquals($expected_page1_last_index, $page1_vars['images'][$items_per_page - 1]['#item_index']);

    // When processing the second page.
    $this->givenTheCurrentPageIs(1);
    $page2_vars = $variables;
    media_gallery_preprocess_media_gallery($page2_vars);

    // Then expect page 2 to have the remaining items in reverse order.
    $expected_page2_count = $total_items - $items_per_page;
    $expected_page2_first_index = $expected_page2_count - 1;
    $expected_page2_last_index = 0;

    $this->assertCount($expected_page2_count, Element::children($page2_vars['images']));
    $this->assertEquals($expected_page2_first_index, $page2_vars['images'][0]['#item_index']);
    $this->assertEquals($expected_page2_last_index, $page2_vars['images'][$expected_page2_count - 1]['#item_index']);
  }

}
