<?php

declare(strict_types=1);

namespace Drupal\media_image_metadata\Service;

use Drupal\Component\Utility\Html;

/**
 * Helper service for extracting image metadata (EXIF, IPTC, XMP).
 */
class ImageMetadataHelper {

  /**
   * Get raw metadata from an image file.
   *
   * @param string $file_path
   *   Path to the image file.
   *
   * @return array
   *   Associative array with keys: 'exif', 'iptc', 'xmp'.
   *
   * @throws \RuntimeException
   *   When the file is not readable.
   */
  public function getMetadataRaw(string $file_path): array {
    $metadata = [
      'exif' => [],
      'iptc' => [],
      'xmp' => [],
    ];

    if (!is_readable($file_path)) {
      throw new \RuntimeException("File not readable: $file_path");
    }

    // EXIF Metadata.
    if (function_exists('exif_read_data')) {
      $exif = @exif_read_data($file_path, 'ANY_TAG', TRUE);
      if ($exif !== FALSE) {
        $metadata['exif'] = $exif;
      }
    }

    // IPTC Metadata.
    $info = [];
    $size = @getimagesize($file_path, $info);
    if (isset($info['APP13'])) {
      $iptc = iptcparse($info['APP13']);
      if ($iptc !== FALSE) {
        foreach ($iptc as $tag => $value) {
          $metadata['iptc'][$tag] = is_array($value) && count($value) === 1 ? $value[0] : $value;
        }
      }
    }

    // XMP Metadata.
    $metadata['xmp'] = $this->extractXmp($file_path);

    return $metadata;
  }

  /**
   * Extract XMP from the image file.
   *
   * @param string $file_path
   *   Path to the image file.
   *
   * @return array
   *   Key-value pairs from XMP data.
   */
  private function extractXmp(string $file_path): array {
    $result = [];
    $data = @file_get_contents($file_path);

    if ($data !== FALSE && preg_match('/<x:xmpmeta.*?<\/x:xmpmeta>/s', $data, $matches)) {
      $xmp = $matches[0];
      try {
        $dom = Html::load($xmp);
        $xpath = new \DOMXPath($dom);

        // Query all elements that don't have child elements (leaf nodes).
        $xpath_query = "//*[not(*)]";
        foreach ($xpath->query($xpath_query) as $element) {
          assert($element instanceof \DOMElement);

          $name = $element->nodeName;
          $value = trim($element->textContent);

          // Skip empty values.
          if ($value === '') {
            continue;
          }

          // Build a hierarchical name if the element has a parent with a
          // different name.
          $parent = $element->parentNode;
          if ($parent instanceof \DOMElement && $parent->nodeName !== $name) {
            $name = $parent->nodeName . ':' . $name;
          }

          $result[$name] = $value;
        }
      }
      catch (\Exception $e) {
        // If XML parsing fails, silently continue.
      }
    }

    return $result;
  }

  /**
   * Normalize metadata into a flat, human-friendly array.
   *
   * @param array $raw
   *   Raw metadata as returned by self::getMetadataRaw().
   *
   * @return array
   *   Normalized metadata.
   */
  public function normalizeMetadata(array $raw): array {
    $iptc = $raw['iptc'];
    $xmp = $raw['xmp'];
    $exif = $raw['exif'];

    return [
      'title' => $iptc['2#005'] ?? $xmp['dc:title'] ?? NULL,
      'alt' => $xmp['rdf:description:photoshop:headline'] ?? $iptc['2#005'] ?? NULL,
      'caption' => $iptc['2#120'] ?? $xmp['dc:description'] ?? NULL,
      'credit' => $iptc['2#110'] ?? NULL,
      'byline' => $iptc['2#080'] ?? NULL,
      'keywords' => $iptc['2#025'] ?? (isset($xmp['dc:subject']) ? explode(',', $xmp['dc:subject']) : NULL),
      'copyright' => $iptc['2#116'] ?? $xmp['dc:rights'] ?? NULL,
      'date_created' => $iptc['2#055'] ?? ($exif['IFD0']['DateTime'] ?? NULL),
      'city' => $iptc['2#090'] ?? NULL,
      'province' => $iptc['2#095'] ?? NULL,
      'country' => $iptc['2#101'] ?? NULL,
      'source' => $iptc['2#115'] ?? NULL,
      'headline' => $iptc['2#105'] ?? NULL,
      'category' => $iptc['2#015'] ?? NULL,
      'supp_cat' => $iptc['2#020'] ?? NULL,
      'urgency' => $iptc['2#010'] ?? NULL,
      // Additional EXIF camera data.
      'camera_model' => $exif['IFD0']['Model'] ?? $exif['EXIF']['Model'] ?? NULL,
      'iso' => $exif['EXIF']['ISOSpeedRatings'] ?? NULL,
      'exposure' => isset($exif['EXIF']['ExposureTime']) ? $this->normalizeFraction($exif['EXIF']['ExposureTime']) : NULL,
      'aperture' => isset($exif['EXIF']['FNumber']) ? $this->normalizeFraction($exif['EXIF']['FNumber']) : NULL,
      'focal_length' => isset($exif['EXIF']['FocalLength']) ? $this->normalizeFraction($exif['EXIF']['FocalLength']) : NULL,
    ];
  }

  /**
   * Normalize fractions.
   *
   * @param string $fraction
   *   The fraction string to normalize (e.g., "1/60", "28/10").
   *
   * @return string|int|float
   *   The normalized fraction value.
   */
  private function normalizeFraction(string $fraction) {
    $parts = explode('/', $fraction);
    $top = (int) $parts[0];
    $bottom = (int) $parts[1];

    if ($top > $bottom) {
      // Value > 1.
      if (($top % $bottom) == 0) {
        $value = ($top / $bottom);
      }
      else {
        $value = round(($top / $bottom), 2);
      }
    }
    else {
      if ($top == $bottom) {
        // Value = 1.
        $value = '1';
      }
      else {
        // Value < 1.
        if ($top == 1) {
          $value = '1/' . $bottom;
        }
        else {
          if ($top != 0) {
            $value = '1/' . round(($bottom / $top), 0);
          }
          else {
            $value = '0';
          }
        }
      }
    }
    return $value;
  }

}
