<?php

namespace Drupal\media_image_style_formatter\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceEntityFormatter;
use Drupal\Core\Security\TrustedCallbackInterface;

class RenderedMediaWithImageStyleFormatter extends EntityReferenceEntityFormatter implements TrustedCallbackInterface {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'view_mode' => 'default',
      'override_image_style' => FALSE,
      'image_style' => '',
      'image_field_name' => 'field_media_image',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    // Get the reference field name.
    $field_name = $this->fieldDefinition->getName();

    // Detectar si estamos en Views.
    $is_views = $this->isViewsContext($form_state);

    // Obtener allowed bundles (con soporte para Views).
    $allowed_bundles = $this->getAllowedBundles();

    // Logging existing media bundles.
    $media_bundles = array_keys(\Drupal::entityTypeManager()->getStorage('media_type')->loadMultiple());

    // Selector for the image field in the media entity, filtered by allowed bundles.
    $image_fields = [];
    $entity_field_manager = \Drupal::service('entity_field.manager');
    $bundles_to_check = empty($allowed_bundles) ? $media_bundles : array_intersect($allowed_bundles, $media_bundles);

    // Get base fields (shared across all bundles), excluding 'thumbnail'.
    $base_fields = $entity_field_manager->getBaseFieldDefinitions('media');
    foreach ($base_fields as $base_field_name => $base_definition) {
      if ($base_definition->getType() === 'image' && $base_field_name !== 'thumbnail') {
        $image_fields[$base_field_name] = $base_definition->getLabel() . ' (' . $base_field_name . ') - Base field (all bundles)';
      }
    }

    // Get instance fields by bundle.
    foreach ($bundles_to_check as $bundle) {
      $fields = $entity_field_manager->getFieldDefinitions('media', $bundle);
      foreach ($fields as $field_name_bundle => $field_definition) {
        if ($field_definition->getType() === 'image' && !$field_definition->getFieldStorageDefinition()->isBaseField()) {
          $image_fields[$field_name_bundle] = $field_definition->getLabel() . ' (' . $field_name_bundle . ') - Bundle: ' . $bundle;
        }
      }
    }

    // Construir opciones de estilo con "None (original image)".
    $style_options = [
      '' => $this->t('- None -'),
      '__original__' => $this->t('None (original image)'),
    ] + image_style_options(FALSE);

    // Add checkbox to enable override.
    $elements['override_image_style'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Override entity image style'),
      '#default_value' => $this->getSetting('override_image_style'),
      '#description' => $this->t('Check this option to ignore the default image style in the media entity\'s view mode and apply a new one here.'),
    ];

    if ($is_views) {
      // ===== CONFIGURACIÓN PARA VIEWS =====
      // Fieldset para agrupar las opciones de override.
      $elements['image_style_override_settings'] = [
        '#type' => 'details',
        '#title' => $this->t('Image style override settings'),
        '#open' => TRUE,
        '#states' => [
          'visible' => [
            ':input[name="options[settings][override_image_style]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $elements['image_style_override_settings']['image_field_name'] = [
        '#type' => 'select',
        '#title' => $this->t('Image field to override'),
        '#options' => $image_fields,
        '#default_value' => $this->getSetting('image_field_name'),
        '#description' => $this->t('Select the image type field in the media entity you want to override.'),
        '#empty_value' => '',
        '#parents' => ['options', 'settings', 'image_field_name'],
      ];

      $elements['image_style_override_settings']['image_style'] = [
        '#title' => $this->t('Image style to apply'),
        '#type' => 'select',
        '#default_value' => $this->getSetting('image_style'),
        '#options' => $style_options,
        '#description' => $this->t('Select an image style to override. "None (original image)" shows the image at its original size.'),
        '#parents' => ['options', 'settings', 'image_style'],
      ];
    }
    else {
      // ===== CONFIGURACIÓN PARA MANAGE DISPLAY (ORIGINAL) =====
      $elements['image_style_settings'] = [
        '#type' => 'details',
        '#title' => $this->t('Image style settings'),
        '#open' => TRUE,
        '#states' => [
          'visible' => [
            ':input[name="fields[' . $field_name . '][settings_edit_form][settings][override_image_style]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $elements['image_style_settings']['image_field_name'] = [
        '#type' => 'select',
        '#title' => $this->t('Image field to override'),
        '#options' => $image_fields,
        '#default_value' => $this->getSetting('image_field_name'),
        '#description' => $this->t('Select the image type field in the media entity you want to override.'),
        '#empty_value' => '',
        '#parents' => ['fields', $field_name, 'settings_edit_form', 'settings', 'image_field_name'],
      ];

      $elements['image_style_settings']['image_style'] = [
        '#title' => $this->t('Image style to apply'),
        '#type' => 'select',
        '#default_value' => $this->getSetting('image_style'),
        '#options' => $style_options,
        '#description' => $this->t('Select an image style to override. "None (original image)" shows the image at its original size.'),
        '#parents' => ['fields', $field_name, 'settings_edit_form', 'settings', 'image_style'],
      ];
    }

    return $elements;
  }

  /**
   * Obtiene los bundles permitidos del campo.
   *
   * @return array
   *   Array de bundles permitidos.
   */
  protected function getAllowedBundles() {
    // Método 1: Desde handler_settings (funciona en Manage Display).
    $handler_settings = $this->fieldDefinition->getSetting('handler_settings');
    if (!empty($handler_settings['target_bundles'])) {
      return $handler_settings['target_bundles'];
    }

    // Método 2: Cargar desde el field config original (funciona en Views).
    $field_name = $this->fieldDefinition->getName();
    $entity_type_id = $this->fieldDefinition->getTargetEntityTypeId();
    
    if ($entity_type_id && $field_name) {
      $field_configs = \Drupal::entityTypeManager()
        ->getStorage('field_config')
        ->loadByProperties([
          'field_name' => $field_name,
          'entity_type' => $entity_type_id,
        ]);

      foreach ($field_configs as $field_config) {
        $settings = $field_config->getSetting('handler_settings');
        if (!empty($settings['target_bundles'])) {
          return $settings['target_bundles'];
        }
      }
    }

    return [];
  }

  /**
   * Detecta si el formulario está siendo usado en contexto de Views.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return bool
   *   TRUE si estamos en Views, FALSE en caso contrario.
   */
  protected function isViewsContext(FormStateInterface $form_state) {
    if ($form_state->has('view')) {
      return TRUE;
    }

    $build_info = $form_state->getBuildInfo();
    $form_id = $build_info['form_id'] ?? '';
    if (strpos($form_id, 'views_ui') !== FALSE) {
      return TRUE;
    }

    $callback = $build_info['callback_object'] ?? NULL;
    if ($callback && strpos(get_class($callback), 'Views') !== FALSE) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $override = $this->getSetting('override_image_style');
    $image_styles = image_style_options(FALSE);
    $image_style = $this->getSetting('image_style');
    $image_field_name = $this->getSetting('image_field_name');

    $summary[] = $this->t('Image field: @field', ['@field' => $image_field_name ?: $this->t('None selected')]);

    if ($override) {
      $summary[] = $this->t('Override image style: Yes');
      if ($image_style === '__original__') {
        $summary[] = $this->t('Applied style: None (original image)');
      }
      elseif (!empty($image_style) && isset($image_styles[$image_style])) {
        $summary[] = $this->t('Applied style: @style', ['@style' => $image_styles[$image_style]]);
      }
      else {
        $summary[] = $this->t('Applied style: None (use default)');
      }
    }
    else {
      $summary[] = $this->t('Override image style: No (use entity default)');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = parent::viewElements($items, $langcode);
    $override = $this->getSetting('override_image_style');
    $image_style = $this->getSetting('image_style');
    $image_field_name = $this->getSetting('image_field_name');
    $field_name = $this->fieldDefinition->getName();

    if ($override && $image_style && $image_field_name) {
      foreach ($elements as $delta => &$build) {
        if (!is_numeric($delta)) {
          continue;
        }

        $build['#image_style_override'] = $image_style;
        $build['#image_field_name'] = $image_field_name;
        $build['#formatter_field_name'] = $field_name;

        if (!isset($build['#cache']['keys'])) {
          $build['#cache']['keys'] = [];
        }
        $build['#cache']['keys'][] = 'media_image_style_formatter';
        $build['#cache']['keys'][] = $field_name;
        $build['#cache']['keys'][] = $image_style;

        $build['#pre_render'][] = [static::class, 'preRenderOverride'];
      }
    }

    return $elements;
  }

  /**
   * Pre-render callback to override the image style.
   */
  public static function preRenderOverride(array $element) {
    $image_style = $element['#image_style_override'] ?? '';
    $image_field_name = $element['#image_field_name'] ?? '';

    if (!$image_style || !$image_field_name || !isset($element[$image_field_name])) {
      return $element;
    }

    $is_original = ($image_style === '__original__');

    foreach ($element[$image_field_name] as $delta => &$image_element) {
      if (!is_numeric($delta)) {
        continue;
      }

      if ($is_original) {
        if (isset($image_element['#theme']) && $image_element['#theme'] === 'image_formatter') {
          $item = $image_element['#item'] ?? NULL;
          if ($item && $item->entity) {
            $uri = $item->entity->getFileUri();
            $image_element = [
              '#theme' => 'image',
              '#uri' => $uri,
              '#alt' => $item->alt ?? '',
              '#title' => $item->title ?? '',
              '#attributes' => $image_element['#item_attributes'] ?? [],
            ];
          }
        }
        elseif (isset($image_element['#theme']) && $image_element['#theme'] === 'image_style') {
          $image_element['#theme'] = 'image';
          unset($image_element['#style_name']);
        }
        if (isset($image_element[0]['#theme']) && $image_element[0]['#theme'] === 'image_style') {
          $image_element[0]['#theme'] = 'image';
          unset($image_element[0]['#style_name']);
        }
      }
      else {
        if (isset($image_element['#theme']) && $image_element['#theme'] === 'image_formatter') {
          $image_element['#image_style'] = $image_style;
          unset($image_element['#cache']);
        }
        if (isset($image_element[0]['#theme']) && $image_element[0]['#theme'] === 'image_style') {
          $image_element[0]['#style_name'] = $image_style;
          unset($image_element[0]['#cache']);
        }
      }
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['preRenderOverride'];
  }

}
