<?php

namespace Drupal\media_kinescope\Plugin\Field\FieldFormatter;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\media\Entity\Media;
use Drupal\media\Entity\MediaType;
use Drupal\media_kinescope\Plugin\media\Source\KinescopeVideo;
use Drupal\media_kinescope\ResourceException;
use Drupal\media_kinescope\ResourceFetcherInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

// cspell:ignore allowtransparency
/**
 * Plugin implementation of the 'kinescope_video' formatter.
 *
 * @FieldFormatter(
 *   id = "kinescope_video",
 *   label = @Translation("Kinescope Video content"),
 *   field_types = {
 *     "link",
 *     "string",
 *     "string_long",
 *   },
 * )
 */
class KinescopeVideoFormatter extends FormatterBase {

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The kinescope.io resource fetcher.
   *
   * @var \Drupal\media_kinescope\ResourceFetcherInterface
   */
  protected $resourceFetcher;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an OEmbedFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin ID for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\media_kinescope\ResourceFetcherInterface $resource_fetcher
   *   The oEmbed resource fetcher service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, MessengerInterface $messenger, ResourceFetcherInterface $resource_fetcher, LoggerChannelFactoryInterface $logger_factory) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->messenger = $messenger;
    $this->resourceFetcher = $resource_fetcher;
    $this->logger = $logger_factory->get('media');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('messenger'),
      $container->get('media_kinescope.resource_fetcher'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $element = [];

    if (!($media = $items->getEntity()) instanceof Media) {
      return $element;
    }

    $configuration = $media->getSource()->getConfiguration();
    $apiKey = $configuration['kinescope_api_key'] ?? NULL;

    foreach ($items as $delta => $item) {
      $main_property = $item->getFieldDefinition()->getFieldStorageDefinition()->getMainPropertyName();
      $value = $item->{$main_property};

      if (empty($value)) {
        continue;
      }

      try {
        /** @var \Drupal\media_kinescope\Resource */
        $resource = $this->resourceFetcher->fetchResource($value, $apiKey);
      }
      catch (ResourceException $exception) {
        $this->logger->error("Could not retrieve the remote URL (@url).", ['@url' => $value]);
        continue;
      }

      $element[$delta] = [
        '#theme' => 'media_kinescope',
        '#url' => $resource->getUrl()->toString(),
        '#attributes' => [
          'width' => $resource->getWidth(),
          'height' => $resource->getHeight(),
          'class' => ['media-kinescope-video-content'],
        ],
        '#thumbnail_url' => $resource->getThumbnailUrl()->toString(),
        '#autoplay' => $configuration['kinescope_autoplay'] == '1' ? '1' : '0'
      ];

      // An empty title attribute will disable title inheritance, so only
      // add it if the resource has a title.
      $title = $resource->getTitle();
      if ($title) {
        $element[$delta]['#attributes']['title'] = $title;
      }

      CacheableMetadata::createFromObject($resource)
        ->addCacheTags([])
        ->applyTo($element[$delta]);
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    if ($field_definition->getTargetEntityTypeId() !== 'media') {
      return FALSE;
    }

    if (parent::isApplicable($field_definition)) {
      $media_type = $field_definition->getTargetBundle();

      if ($media_type) {
        $media_type = MediaType::load($media_type);
        return $media_type && $media_type->getSource() instanceof KinescopeVideo;
      }
    }
    return FALSE;
  }

}
