<?php

namespace Drupal\media_kinescope;

use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Cache\CacheableDependencyTrait;
use Drupal\Core\Url;

/**
 * Value object representing an kinescope.io resource.
 */
class Resource implements CacheableDependencyInterface {

  use CacheableDependencyTrait;

  /**
   * A text title, describing the resource.
   *
   * @var string
   */
  protected $title;

  /**
   * A text subtitle, describing the resource.
   *
   * @var string
   */
  protected $subtitle;

  /**
   * A text decription, describing the resource.
   *
   * @var string
   */
  protected $description;

  /**
   * A URL to a thumbnail image representing the resource.
   *
   * @var string
   */
  protected $thumbnailUrl;

  /**
   * The resource URL.
   *
   * @var string
   */
  protected $embedUrl;

  /**
   * The resource resolution.
   *
   * @var string
   */
  protected $resolution;

  /**
   * The width of the resource, in pixels.
   *
   * @var int
   */
  protected $width;

  /**
   * The height of the resource, in pixels.
   *
   * @var int
   */
  protected $height;

  /**
   * Resource constructor.
   *
   * @param string $title
   *   (optional) A text title, describing the resource.
   * @param string $subtitle
   *   (optional) A text subtitle, describing the resource.
   * @param string $description
   *   (optional) A text, describing the resource.
   * @param string $thumbnail_url
   *   (optional) A URL to a thumbnail image representing the resource.
   * @param string $embed_url
   *   (optional) A URL to the resource.
   * @param string $resolution
   *   (optional) A resolution of the resource.
   */
  public function __construct($title = NULL, $subtitle = NULL, $description = NULL, $thumbnail_url = NULL, $embed_url = NULL, $resolution = NULL) {
    $this->title = $title;
    $this->subtitle = $subtitle;
    $this->description = $description;
    if ($thumbnail_url) {
      $this->thumbnailUrl = $thumbnail_url;
    }
    if ($embed_url) {
      $this->embedUrl = $embed_url;
    }
    if ($resolution) {
      $this->resolution = $resolution;
      [$this->width, $this->height] = explode("x", $resolution);
    }

  }

  /**
   * Sets the dimensions.
   *
   * @param int|null $width
   *   The width of the resource.
   * @param int|null $height
   *   The height of the resource.
   *
   * @throws \InvalidArgumentException
   *   If either $width or $height are not numbers greater than zero.
   */
  protected function setDimensions($width, $height)
  {
    if ((isset($width) && $width <= 0) || (isset($height) && $height <= 0)) {
      throw new \InvalidArgumentException('The dimensions must be NULL or numbers greater than zero.');
    }
    $this->width = isset($width) ? (int) $width : NULL;
    $this->height = isset($height) ? (int) $height : NULL;
  }

  /**
   * Returns the title of the resource.
   *
   * @return string|null
   *   The title of the resource, if known.
   */
  public function getTitle() {
    return $this->title;
  }

  /**
   * Returns the subtitle of the resource.
   *
   * @return string|null
   *   The subtitle of the resource, if known.
   */
  public function getSubtitle()
  {
    return $this->subtitle;
  }

  /**
   * Returns the description of the resource.
   *
   * @return string|null
   *   The description of the resource, if known.
   */
  public function getDescription()
  {
    return $this->description;
  }

  /**
   * Returns the URL of the resource's thumbnail image.
   *
   * @return \Drupal\Core\Url|null
   *   The absolute URL of the thumbnail image, or NULL if there isn't one.
   */
  public function getThumbnailUrl() {
    return $this->thumbnailUrl ? Url::fromUri($this->thumbnailUrl)->setAbsolute() : NULL;
  }

  /**
   * Returns the embedding URL of the resource.
   *
   * @return \Drupal\Core\Url|null
   *   The resource URL, if it has one.
   */
  public function getUrl() {
    if ($this->embedUrl) {
      return Url::fromUri($this->embedUrl)->setAbsolute();
    }
    return NULL;
  }

  /**
   * Returns the width of the resource.
   *
   * @return int|null
   *   The width of the resource in pixels, or NULL if the resource has no
   *   width.
   */
  public function getWidth()
  {
    return $this->width;
  }

  /**
   * Returns the height of the resource.
   *
   * @return int|null
   *   The height of the resource in pixels, or NULL if the resource has no
   *   height.
   */
  public function getHeight()
  {
    return $this->height;
  }

}
