<?php

namespace Drupal\media_kinescope;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\TransferException;
use GuzzleHttp\RequestOptions;

/**
 * Fetches and caches oEmbed resources.
 */
class ResourceFetcher implements ResourceFetcherInterface {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cacheBackend;

  /**
   * Constructs a ResourceFetcher object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   The cache backend.
   */
  public function __construct(ClientInterface $http_client, CacheBackendInterface $cache_backend) {
    $this->httpClient = $http_client;
    $this->cacheBackend = $cache_backend;
  }

  /**
   * {@inheritdoc}
   */
  public function fetchResource($url, $apiKey) {
    $cache_id = "media:kinescope_video_resource:$url";

    $cached = $this->cacheBackend->get($cache_id);
    if ($cached) {
      return $this->createResource($cached->data, $url);
    }

    $matches = [];
    preg_match('/([^\/]+$)/', $url, $matches);
    $apiurl = 'https://api.kinescope.io/v1/videos/' . $matches[0];

    try {
      $response = $this->httpClient->request('GET', $apiurl, [
        RequestOptions::TIMEOUT => 5,
        RequestOptions::HEADERS => [
          'Content-Type' => 'application/json',
          'Authorization' => 'Bearer ' . $apiKey,
        ],
      ]);
    }
    catch (TransferException $e) {
      throw new ResourceException('Could not retrieve the kinescope.io resource.', $url, [], $e);
    }

    [$format] = $response->getHeader('Content-Type');
    $content = (string) $response->getBody();

    if (strstr($format, 'application/json')) {
      $data = Json::decode($content);
      if (json_last_error() !== JSON_ERROR_NONE) {
        throw new ResourceException('Error decoding kinescope.io resource: ' . json_last_error_msg(), $url);
      }
    }
    if (empty($data) || !is_array($data) || empty($data['data']) || !is_array($data['data'])) {
      throw new ResourceException('The kinescope.io resource could not be decoded.', $url);
    }

    $this->cacheBackend->set($cache_id, $data['data']);

    return $this->createResource($data['data'], $url);
  }

  /**
   * Creates a Resource object from raw resource data.
   *
   * @param array $data
   *   The resource data returned by the provider.
   * @param string $url
   *   The URL of the resource.
   *
   * @return \Drupal\media_kinescope\Resource
   *   A value object representing the resource.
   *
   * @throws \Drupal\media_kinescope\ResourceException
   *   If the resource cannot be created.
   */
  protected function createResource(array $data, $url) {
    $data += [
      'title' => NULL,
      'subtitle' => NULL,
      'description' => NULL,
      'thumbnail_url' => NULL,
      'embed_link' => NULL,
      'resolution' => NULL,
    ];

    if (isset($data['poster']) && isset($data['poster']['original'])) {
      $data['thumbnail_url'] = $data['poster']['original'];
    }

    if (isset($data['assets']) && isset($data['assets'][0]) && isset($data['assets'][0]['resolution'])) {
      $data['resolution'] = $data['assets'][0]['resolution'];
    }

    try {
      return new Resource(
        $data['title'],
        $data['subtitle'],
        $data['description'],
        $data['thumbnail_url'],
        $data['embed_link'],
        $data['resolution']
      );
    }
    catch (\InvalidArgumentException $e) {
      throw new ResourceException($e->getMessage(), $url, $data, $e);
    }
  }

}
