<?php

namespace Drupal\media_thumbnails_excel\Plugin\MediaThumbnail;

use Drupal\media_thumbnails\Plugin\MediaThumbnailBase;

use PhpOffice\PhpSpreadsheet\IOFactory as SpreadsheetIOFactory;

/**
 * Media thumbnail plugin for Excel documents.
 *
 * @MediaThumbnail(
 *   id = "media_thumbnails_excel",
 *   label = @Translation("Media Thumbnail Excel"),
 *   mime = {
 *     "application/vnd.ms-excel",
 *     "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
 *   }
 * )
 */
class MediaThumbnailsExcel extends MediaThumbnailBase {

  /**
   * Creates a managed thumbnail file using the passed source file uri.
   *
   * {@inheritdoc}
   */
  public function createThumbnail($sourceUri) {

    // Check the required php extension.
    if (!extension_loaded('imagick')) {
      $this->logger->warning($this->t('Imagick php extension not loaded.'));
      return NULL;
    }
    //src : https://stackoverflow.com/questions/7708534/create-a-thumbnail-preview-of-documents-pdf-doc-xls-etc-in-php-lamp
    $pathPdf = NULL;

    // Imagick doesn't support stream wrappers!
    $path = $this->fileSystem->realpath($sourceUri);

    $ext = pathinfo($sourceUri, PATHINFO_EXTENSION); // To get extension

    // Read the pdf.
    $im = new \Imagick();

    if ($ext == 'xls' || $ext == 'xlsx') {
      $spreadsheet = SpreadsheetIOFactory::load($path);
      $spreadsheet->getActiveSheet()->getParent()->getDefaultStyle()->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID);
      $spreadsheet->getActiveSheet()->getParent()->getDefaultStyle()->getFill()->getStartColor()->setARGB('FFFFFFFF');

      // Create a new PDF writer using mPDF
      // src: https://phpspreadsheet.readthedocs.io/en/latest/topics/reading-and-writing-to-file/#pdf
      $class = \PhpOffice\PhpSpreadsheet\Writer\Pdf\Mpdf::class;
      \PhpOffice\PhpSpreadsheet\IOFactory::registerWriter('Pdf', $class);
      $writer = \PhpOffice\PhpSpreadsheet\IOFactory::createWriter($spreadsheet, 'Pdf');

      // Set the output file path
      $pathPdf = $sourceUri . '.pdf';

      // Write the PDF to the output file path
      $writer->save($pathPdf);
    }

    // Imagick doesn't support stream wrappers!
    $pathPdf = $this->fileSystem->realpath($pathPdf);

    try {
      $im->readimage($pathPdf . '[0]');
    }
    catch (\ImagickException $e) {
      $this->logger->warning($e->getMessage() . ' $pathPdf = ' . $pathPdf);
      return NULL;
    }

    // Handle transparency stuff.
    $im->setImageBackgroundColor('white');
    $im->setImageAlphaChannel(\Imagick::ALPHACHANNEL_REMOVE);
    try {
      $im->mergeImageLayers(\Imagick::LAYERMETHOD_FLATTEN);
    }
    catch (\ImagickException $e) {
      $this->logger->warning($e->getMessage());
      return NULL;
    }

    // Resize the thumbnail to the globally configured width.
    $width = $this->configuration['width'] ?? 500;
    if ($im->getImageWidth() > $width) {
      try {
        $im->scaleImage($width, 0);
      }
      catch (\ImagickException $e) {
        $this->logger->warning($e->getMessage());
        return NULL;
      }
    }

    // Convert the image to JPG.
    $im->setImageFormat('jpg');
    $image = $im->getImageBlob();
    $im->clear();
    $im->destroy();

    // Return a new managed file object using the generated thumbnail.
    return \Drupal::service('file.repository')->writeData($image, $sourceUri . '.jpg');
  }

}
