<?php

namespace Drupal\mediaflow\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Controller\ControllerBase;
use Drupal\field\Entity\FieldConfig;
use Drupal\mediaflow\Service\UsageManager;
use Drupal\mediaflow\Service\UsageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Implements Mediaflow controller.
 */
class MediaflowController extends ControllerBase {

  /**
   * The usage manager.
   *
   * @var \Drupal\mediaflow\Service\UsageManager
   */
  private $usageManager;

  /**
   * Constructs an MediaflowController instance.
   *
   * @param \Drupal\mediaflow\Service\UsageManager $usage_manager
   *   The usage manager.
   */
  public function __construct(UsageManager $usage_manager) {
    $this->usageManager = $usage_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('mediaflow.usage_manager')
    );
  }

  /**
   * Retrieves a short-lived access token.
   */
  public function token(Request $request) {
    $token = $this->usageManager->getFetcher()->getAccessToken();
    return new JsonResponse($token);
  }

  /**
   * Add media from data.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Response.
   */
  public function addMedia(Request $request) {
    $data = Json::decode($request->get('mediaflow-info'));
    $upload_image = $this->usageManager->getFetcher()->downloadFile($data);

    $url = $uuid = '';
    if ($upload_image) {
      $storage = $this->entityTypeManager()->getStorage('media');
      $media = $storage->create([
        'name' => $data['name'],
        'bundle' => $data['bundle'] ?? $data['basetype'],
        'uid' => $this->currentUser()->id(),
        'status' => 1,
      ]);
      foreach ($media->getFieldDefinitions() as $field) {
        if ($field->getType() == 'image' && $field instanceof FieldConfig) {
          $media->set($field->getName(), [
            'target_id' => $upload_image->id(),
            'alt' => $data['altText'],
            'title' => $data['name'],
          ]);
          break;
        }
      }
      $media->save();
      $uuid = $media->uuid();
      $url = $upload_image->createFileUrl();

      $values = [
        'mediaflow_origin_id' => $data['id'],
        'file' => $upload_image->id(),
        'type' => UsageManagerInterface::FILE,
        'data' => serialize($data),
      ];
      $this->usageManager->saveMediaData($values);
    }

    return new JsonResponse([
      'src' => $url,
      'data-entity-uuid' => $uuid,
      'data-entity-type' => 'media',
      'alt' => $data['altText'],
    ]);
  }

}
