<?php

namespace Drupal\mediaflow\Element;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\FormElementBase;
use Drupal\Core\Url;

/**
 * Provides a Mediaflow resource selector form element.
 *
 * Usage example:
 *
 * @code
 * $build = [
 *   '#type' => 'mediaflow_selector_file',
 * ];
 * @endcode
 *
 * @FormElement("mediaflow_selector_file")
 */
class MediaflowSelectorFile extends FormElementBase {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $class = get_class($this);

    return [
      '#process' => [
        [$class, 'processMediaflowSelector'],
      ],
      '#element_validate' => [
        [$class, 'validateMediaflowSelector'],
      ],
    ];
  }

  /**
   * Process Mediaflow Selector element.
   *
   * @param array $element
   *   The render element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param array $complete_form
   *   The form array.
   */
  public static function processMediaflowSelector(array &$element, FormStateInterface $form_state, array &$complete_form) {
    $settings = \Drupal::config('mediaflow.settings');
    $element['#tree'] = TRUE;

    $element['value'] = [
      '#type' => 'hidden',
      '#attributes' => [
        'data-mediaflow-target' => TRUE,
      ],
    ];

    if (is_numeric($element['#value'])) {
      $element['value']['#default_value'] = $element['#value'];
    }
    else {
      if (strlen($element['#value']) == 0) {
        $element['value']['#default_value'] = "0";
      }
    }

    $limit_file_type = 'image,video,srt,vtt';
    if (!empty($element['#media_type'])) {
      $limit_file_type = $element['#media_type'];
    }

    $api_settings = [
      'tokenEndpoint' => Url::fromRoute('mediaflow.token')->toString(),
      'allowSelectFormat' => $settings->get('method'),
      'forceAltText' => $settings->get('set_alt_text'),
      'locale' => \Drupal::currentUser()->getPreferredAdminLangcode() != 'sv' ? 'en_US' : 'sv_SE',
      'limitFileType' => $limit_file_type,
    ];

    if (!empty($element['#value'])) {
      $element['mediaflowbrowser'] = [
        '#type' => 'markup',
        '#markup' => '<h5>' . $element['#title'] . '</h5><span class="button pick-mediaflow-button" data-drupal-selector="edit-submit">Pick Mediaflow media</span><span class="remove-mediaflow-button button button--danger">Remove media</span><div class="mediaflow-file" style="position:relative;width:1000px;height:500px;overflow:hidden;border:1px solid black;margin:10px; display: none;" data-mediaflow-selector-file></div>',
        '#attached' => [
          'library' => ['mediaflow/element_selector'],
          'drupalSettings' => ['mediaflow' => $api_settings],
        ],
      ];
    }
    else {
      $element['mediaflowbrowser'] = [
        '#type' => 'markup',
        '#markup' => '<h5>' . $element['#title'] . '</h5><span class="button pick-mediaflow-button" data-drupal-selector="edit-submit">Pick Mediaflow media</span><div class="mediaflow-file" style="position:relative;width:1000px;height:500px;overflow:hidden;border:1px solid black;margin:10px; display: none;" data-mediaflow-selector-file></div>',
        '#attached' => [
          'library' => ['mediaflow/element_selector'],
          'drupalSettings' => ['mediaflow' => $api_settings],
        ],
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function valueCallback(&$element, $input, FormStateInterface $form_state) {
    if ($input === FALSE) {
      $element += ['#default_value' => ''];
      return $element['#default_value'];
    }

    if (!empty($input['value']) && $input['value'] != "0") {
      $data = Json::decode($input['value']);

      if (is_array($data)) {
        if (!$form_state->hasTemporaryValue('filename') || $form_state->getTemporaryValue('filename') != $data['filename']) {
          $returnData = \Drupal::service('mediaflow.usage_manager')->addMedia($data);
          $form_state->setTemporaryValue('filename', $data['filename']);
          $form_state->setTemporaryValue('data', $returnData);
          return $returnData;
        }
        else {
          return $form_state->getTemporaryValue('data');
        }
      }
    }
    elseif ($input['value'] == "0") {
      return "";
    }

    return "";
  }

  /**
   * Validate Mediaflow Selector element.
   *
   * @param array $element
   *   The render element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param array $complete_form
   *   The form array.
   */
  public static function validateMediaflowSelector(array &$element, FormStateInterface $form_state, array &$complete_form) {
    $input = $form_state->getValue($element['#parents']);

    // Check to see if old id, new json or reset.
    if (!empty($input['value']) && $input['value'] != "0") {
      if (is_numeric($input['value'])) {
        $form_state->setValueForElement($element, $input['value']);
      }
      elseif (!$form_state->hasAnyErrors()) {
        $data = Json::decode($input['value']);

        if (!$form_state->hasTemporaryValue('filename') || $form_state->getTemporaryValue('filename') != $data['filename']) {
          $returnData = \Drupal::service('mediaflow.usage_manager')->addMedia($data);
          $form_state->setTemporaryValue('filename', $data['filename']);
          $form_state->setTemporaryValue('data', $returnData);
          $mediaflow_id = $returnData;
        }
        else {
          $mediaflow_id = $form_state->getTemporaryValue('data');
        }

        if (!empty($mediaflow_id)) {
          $form_state->setValueForElement($element, $mediaflow_id);
        }
      }
    }
    elseif ($input['value'] == "0") {
      $form_state->setValueForElement($element, "");
    }
  }

}
